<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class R2CC_Admin {

    public function __construct() {
        add_action( 'admin_menu', array( $this, 'add_menu' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_init', array( $this, 'handle_export' ) );
        add_filter( 'plugin_action_links_' . R2CC_PLUGIN_BASENAME, array( $this, 'action_links' ) );
    }

    public function add_menu() {
        add_menu_page(
            __( 'Cookie Consent', 'rise2-cookie-consent' ),
            __( 'Cookie Consent', 'rise2-cookie-consent' ),
            'manage_options',
            'r2cc-settings',
            array( $this, 'render_page' ),
            'dashicons-shield',
            81
        );
    }

    public function action_links( $links ) {
        $settings_link = '<a href="' . admin_url( 'admin.php?page=r2cc-settings' ) . '">' . __( 'Settings', 'rise2-cookie-consent' ) . '</a>';
        array_unshift( $links, $settings_link );
        return $links;
    }

    public function enqueue_assets( $hook ) {
        if ( strpos( $hook, 'r2cc-settings' ) === false ) return;

        wp_enqueue_style( 'wp-color-picker' );
        wp_enqueue_script( 'wp-color-picker' );

        wp_enqueue_style(
            'r2cc-admin',
            R2CC_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            R2CC_VERSION
        );

        wp_enqueue_script(
            'r2cc-admin',
            R2CC_PLUGIN_URL . 'assets/js/admin.js',
            array( 'jquery', 'wp-color-picker' ),
            R2CC_VERSION,
            true
        );

        wp_localize_script( 'r2cc-admin', 'r2ccAdmin', array(
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'r2cc_admin_nonce' ),
            'siteUrl' => home_url( '/' ),
            'strings' => array(
                'scanComplete'  => __( 'Scan complete!', 'rise2-cookie-consent' ),
                'confirmDelete' => __( 'Are you sure you want to delete this cookie?', 'rise2-cookie-consent' ),
                'saved'         => __( 'Settings saved.', 'rise2-cookie-consent' ),
                'scanning'      => __( 'Scanning...', 'rise2-cookie-consent' ),
            ),
        ) );
    }

    public function register_settings() {
        // General tab fields
        $general_fields = array(
            'enabled', 'banner_position', 'consent_expiry_days',
            'show_reject_all', 'show_powered_by', 'auto_block_scripts',
            'banner_title', 'banner_description',
            'btn_accept_label', 'btn_reject_label', 'btn_settings_label', 'btn_save_label',
            'privacy_policy_url', 'privacy_policy_label', 'geo_restriction',
            'consent_version', 'scan_frequency', 'scan_email_notify',
            // Privacy Policy Generator fields
            'pp_company_name', 'pp_company_address', 'pp_company_oib', 'pp_dpo_email',
        );

        // Banner & Design tab fields
        $banner_fields = array(
            'banner_bg_color', 'banner_text_color',
            'btn_accept_bg', 'btn_accept_text', 'btn_reject_bg', 'btn_reject_text',
            'btn_settings_bg', 'btn_settings_text', 'custom_css',
        );

        // Google Consent Mode tab fields
        $gcm_fields = array(
            'gcm_enabled', 'gcm_wait_for_update', 'gcm_ads_data_redaction',
            'gcm_url_passthrough', 'gcm_region',
        );

        foreach ( $general_fields as $field ) {
            register_setting( 'r2cc_settings_general', 'r2cc_' . $field );
        }
        foreach ( $banner_fields as $field ) {
            register_setting( 'r2cc_settings_banner', 'r2cc_' . $field );
        }
        foreach ( $gcm_fields as $field ) {
            register_setting( 'r2cc_settings_gcm', 'r2cc_' . $field );
        }

        // Handle scan frequency change
        add_action( 'update_option_r2cc_scan_frequency', function( $old, $new ) {
            R2CC_Cron::schedule( $new );
        }, 10, 2 );
    }

    public function handle_export() {
        if ( isset( $_GET['r2cc_export_csv'] ) && current_user_can( 'manage_options' ) ) {
            check_admin_referer( 'r2cc_export' );
            R2CC_Consent_Logger::export_csv();
        }
    }

    public function render_page() {
        $active_tab = isset( $_GET['tab'] ) ? sanitize_text_field( $_GET['tab'] ) : 'general';
        $tabs = array(
            'dashboard' => __( 'Dashboard', 'rise2-cookie-consent' ),
            'general'  => __( 'General', 'rise2-cookie-consent' ),
            'banner'   => __( 'Banner & Design', 'rise2-cookie-consent' ),
            'cookies'  => __( 'Cookies', 'rise2-cookie-consent' ),
            'scanner'  => __( 'Scanner', 'rise2-cookie-consent' ),
            'gcm'      => __( 'Google Consent Mode', 'rise2-cookie-consent' ),
            'privacy'  => __( 'Privacy Policy', 'rise2-cookie-consent' ),
            'tools'    => __( 'Tools', 'rise2-cookie-consent' ),
            'consent'  => __( 'Consent Log', 'rise2-cookie-consent' ),
        );
        ?>
        <div class="wrap r2cc-admin-wrap">
            <div class="r2cc-admin-header">
                <div class="r2cc-admin-logo">
                    <svg width="32" height="32" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                        <path d="M12 2C6.477 2 2 6.477 2 12s4.477 10 10 10 10-4.477 10-10c0-.34-.018-.676-.053-1.008a3.5 3.5 0 01-2.447-.96 3.5 3.5 0 01-.96-2.447A3.494 3.494 0 0116.585 5.5a3.5 3.5 0 01-.632-3.482A10.054 10.054 0 0012 2z" fill="#4f46e5" opacity="0.15"/>
                        <path d="M12 2C6.477 2 2 6.477 2 12s4.477 10 10 10 10-4.477 10-10c0-.34-.018-.676-.053-1.008a3.5 3.5 0 01-2.447-.96 3.5 3.5 0 01-.96-2.447A3.494 3.494 0 0116.585 5.5a3.5 3.5 0 01-.632-3.482A10.054 10.054 0 0012 2z" stroke="#4f46e5" stroke-width="1.5"/>
                        <circle cx="8.5" cy="10.5" r="1.5" fill="#4f46e5"/><circle cx="11" cy="15" r="1.5" fill="#4f46e5"/><circle cx="15" cy="11" r="1" fill="#4f46e5"/>
                    </svg>
                    <div>
                        <h1><?php esc_html_e( 'RISE2 Cookie Consent', 'rise2-cookie-consent' ); ?></h1>
                        <span class="r2cc-version">v<?php echo esc_html( R2CC_VERSION ); ?></span>
                    </div>
                </div>
            </div>

            <nav class="r2cc-tabs">
                <?php foreach ( $tabs as $slug => $label ) :
                    $url = add_query_arg( array( 'page' => 'r2cc-settings', 'tab' => $slug ), admin_url( 'admin.php' ) );
                    $class = $active_tab === $slug ? 'r2cc-tab active' : 'r2cc-tab';
                    ?>
                    <a href="<?php echo esc_url( $url ); ?>" class="<?php echo esc_attr( $class ); ?>"><?php echo esc_html( $label ); ?></a>
                <?php endforeach; ?>
            </nav>

            <div class="r2cc-tab-content">
                <?php
                switch ( $active_tab ) {
                    case 'general':
                        $this->render_tab_general();
                        break;
                    case 'banner':
                        $this->render_tab_banner();
                        break;
                    case 'cookies':
                        $this->render_tab_cookies();
                        break;
                    case 'scanner':
                        $this->render_tab_scanner();
                        break;
                    case 'gcm':
                        $this->render_tab_gcm();
                        break;
                    case 'privacy':
                        $this->render_tab_privacy_policy();
                        break;
                    case 'tools':
                        $this->render_tab_tools();
                        break;
                    case 'consent':
                        $this->render_tab_consent();
                        break;
                    default:
                        $this->render_tab_dashboard();
                        break;
                }
                ?>
            </div>
        </div>
        <?php
    }

    /* ============================================================
       TAB: Dashboard (Analytics)
       ============================================================ */
    private function render_tab_dashboard() {
        $stats = R2CC_Consent_Logger::get_stats();
        $type_breakdown = R2CC_Consent_Logger::get_type_breakdown();
        $category_rates = R2CC_Consent_Logger::get_category_rates();
        $last_scan = R2CC_Cron::get_last_scan();
        ?>
        <div class="r2cc-dashboard">

            <!-- Quick Stats -->
            <div class="r2cc-stats-grid r2cc-stats-grid-4">
                <div class="r2cc-stat-card">
                    <div class="r2cc-stat-number"><?php echo intval( $stats['total'] ); ?></div>
                    <div class="r2cc-stat-label"><?php esc_html_e( 'Total Consents', 'rise2-cookie-consent' ); ?></div>
                </div>
                <div class="r2cc-stat-card">
                    <div class="r2cc-stat-number"><?php echo intval( $stats['last_30_days'] ); ?></div>
                    <div class="r2cc-stat-label"><?php esc_html_e( 'Last 30 Days', 'rise2-cookie-consent' ); ?></div>
                </div>
                <div class="r2cc-stat-card r2cc-stat-accept">
                    <div class="r2cc-stat-number"><?php echo isset( $type_breakdown['accept_all'] ) ? esc_html( $type_breakdown['accept_all']['rate'] . '%' ) : '0%'; ?></div>
                    <div class="r2cc-stat-label"><?php esc_html_e( 'Accept All Rate', 'rise2-cookie-consent' ); ?></div>
                </div>
                <div class="r2cc-stat-card r2cc-stat-reject">
                    <div class="r2cc-stat-number"><?php echo isset( $type_breakdown['reject_all'] ) ? esc_html( $type_breakdown['reject_all']['rate'] . '%' ) : '0%'; ?></div>
                    <div class="r2cc-stat-label"><?php esc_html_e( 'Reject All Rate', 'rise2-cookie-consent' ); ?></div>
                </div>
            </div>

            <div class="r2cc-dashboard-row">
                <!-- Consent Trend Chart -->
                <div class="r2cc-card r2cc-card-wide">
                    <div class="r2cc-card-header-row">
                        <h2 class="r2cc-card-title"><?php esc_html_e( 'Consent Activity (Last 30 Days)', 'rise2-cookie-consent' ); ?></h2>
                        <select id="r2cc-chart-range" class="r2cc-select" style="width:auto;">
                            <option value="7"><?php esc_html_e( '7 days', 'rise2-cookie-consent' ); ?></option>
                            <option value="30" selected><?php esc_html_e( '30 days', 'rise2-cookie-consent' ); ?></option>
                            <option value="90"><?php esc_html_e( '90 days', 'rise2-cookie-consent' ); ?></option>
                        </select>
                    </div>
                    <div id="r2cc-chart-container" style="position:relative;height:280px;margin-top:16px;">
                        <canvas id="r2cc-consent-chart"></canvas>
                    </div>
                </div>

                <!-- Category Acceptance Rates -->
                <div class="r2cc-card">
                    <h2 class="r2cc-card-title"><?php esc_html_e( 'Category Acceptance Rates', 'rise2-cookie-consent' ); ?></h2>
                    <div class="r2cc-cat-rates" style="margin-top:16px;">
                        <?php
                        $cat_colors = array(
                            'necessary'  => '#16a34a',
                            'functional' => '#2563eb',
                            'analytics'  => '#d97706',
                            'marketing'  => '#dc2626',
                        );
                        $cat_labels = array(
                            'necessary'  => __( 'Necessary', 'rise2-cookie-consent' ),
                            'functional' => __( 'Functional', 'rise2-cookie-consent' ),
                            'analytics'  => __( 'Analytics', 'rise2-cookie-consent' ),
                            'marketing'  => __( 'Marketing', 'rise2-cookie-consent' ),
                        );
                        foreach ( $cat_labels as $cat => $label ) :
                            $rate = isset( $category_rates[ $cat ] ) ? $category_rates[ $cat ]['rate'] : 0;
                            $color = $cat_colors[ $cat ];
                            ?>
                            <div class="r2cc-rate-bar-wrap" style="margin-bottom:14px;">
                                <div style="display:flex;justify-content:space-between;margin-bottom:4px;">
                                    <span style="font-weight:600;font-size:13px;"><?php echo esc_html( $label ); ?></span>
                                    <span style="font-weight:600;font-size:13px;color:<?php echo esc_attr( $color ); ?>;"><?php echo esc_html( $rate ); ?>%</span>
                                </div>
                                <div style="background:#e5e7eb;border-radius:6px;height:10px;overflow:hidden;">
                                    <div style="background:<?php echo esc_attr( $color ); ?>;height:100%;border-radius:6px;width:<?php echo esc_attr( $rate ); ?>%;transition:width 0.5s;"></div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

            <div class="r2cc-dashboard-row">
                <!-- Consent Type Breakdown -->
                <div class="r2cc-card">
                    <h2 class="r2cc-card-title"><?php esc_html_e( 'Consent Type Breakdown', 'rise2-cookie-consent' ); ?></h2>
                    <div id="r2cc-type-breakdown" style="margin-top:16px;">
                        <?php if ( empty( $type_breakdown ) ) : ?>
                            <p style="color:#999;"><?php esc_html_e( 'No consent data yet.', 'rise2-cookie-consent' ); ?></p>
                        <?php else : ?>
                            <?php
                            $type_colors = array(
                                'accept_all' => '#16a34a',
                                'reject_all' => '#dc2626',
                                'custom'     => '#2563eb',
                            );
                            $type_labels = array(
                                'accept_all' => __( 'Accept All', 'rise2-cookie-consent' ),
                                'reject_all' => __( 'Reject All', 'rise2-cookie-consent' ),
                                'custom'     => __( 'Custom', 'rise2-cookie-consent' ),
                            );
                            foreach ( $type_breakdown as $type => $data ) :
                                $color = $type_colors[ $type ] ?? '#6b7280';
                                $label = $type_labels[ $type ] ?? ucwords( str_replace( '_', ' ', $type ) );
                                ?>
                                <div style="display:flex;align-items:center;margin-bottom:12px;">
                                    <span style="width:12px;height:12px;border-radius:3px;background:<?php echo esc_attr( $color ); ?>;margin-right:10px;flex-shrink:0;"></span>
                                    <span style="flex:1;font-size:13px;"><?php echo esc_html( $label ); ?></span>
                                    <span style="font-weight:700;font-size:14px;"><?php echo esc_html( $data['count'] ); ?></span>
                                    <span style="color:#888;font-size:12px;margin-left:6px;width:45px;text-align:right;">(<?php echo esc_html( $data['rate'] ); ?>%)</span>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Quick Info -->
                <div class="r2cc-card">
                    <h2 class="r2cc-card-title"><?php esc_html_e( 'Status', 'rise2-cookie-consent' ); ?></h2>
                    <table class="r2cc-status-table" style="margin-top:12px;">
                        <tr>
                            <td><?php esc_html_e( 'Banner', 'rise2-cookie-consent' ); ?></td>
                            <td>
                                <?php if ( get_option( 'r2cc_enabled', 1 ) ) : ?>
                                    <span style="color:#16a34a;font-weight:600;">● <?php esc_html_e( 'Active', 'rise2-cookie-consent' ); ?></span>
                                <?php else : ?>
                                    <span style="color:#dc2626;">● <?php esc_html_e( 'Disabled', 'rise2-cookie-consent' ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e( 'Auto-blocker', 'rise2-cookie-consent' ); ?></td>
                            <td>
                                <?php if ( get_option( 'r2cc_auto_block_scripts', 0 ) ) : ?>
                                    <span style="color:#16a34a;font-weight:600;">● <?php esc_html_e( 'Active', 'rise2-cookie-consent' ); ?></span>
                                <?php else : ?>
                                    <span style="color:#888;">● <?php esc_html_e( 'Off', 'rise2-cookie-consent' ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e( 'Google Consent Mode', 'rise2-cookie-consent' ); ?></td>
                            <td>
                                <?php if ( get_option( 'r2cc_gcm_enabled', 0 ) ) : ?>
                                    <span style="color:#16a34a;font-weight:600;">● v2</span>
                                <?php else : ?>
                                    <span style="color:#888;">● <?php esc_html_e( 'Off', 'rise2-cookie-consent' ); ?></span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e( 'Consent Version', 'rise2-cookie-consent' ); ?></td>
                            <td><code><?php echo esc_html( get_option( 'r2cc_consent_version', '1.0' ) ); ?></code></td>
                        </tr>
                        <tr>
                            <td><?php esc_html_e( 'Cookies Registered', 'rise2-cookie-consent' ); ?></td>
                            <td><strong><?php
                                global $wpdb;
                                echo intval( $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->prefix}r2cc_cookies" ) );
                            ?></strong></td>
                        </tr>
                        <?php if ( $last_scan ) : ?>
                        <tr>
                            <td><?php esc_html_e( 'Last Scan', 'rise2-cookie-consent' ); ?></td>
                            <td><?php echo esc_html( $last_scan['timestamp'] ); ?> (<?php echo intval( $last_scan['detected'] ); ?> <?php esc_html_e( 'found', 'rise2-cookie-consent' ); ?>)</td>
                        </tr>
                        <?php endif; ?>
                    </table>
                </div>
            </div>

        </div>
        <?php
    }

    /* ============================================================
       TAB: Tools (Import/Export, Shortcodes)
       ============================================================ */
    private function render_tab_tools() {
        ?>
        <!-- Import / Export -->
        <div class="r2cc-card">
            <h2 class="r2cc-card-title"><?php esc_html_e( 'Import / Export Settings', 'rise2-cookie-consent' ); ?></h2>
            <p class="r2cc-card-desc"><?php esc_html_e( 'Export your full configuration (settings + cookie database) as JSON. Import to quickly set up another site with the same configuration.', 'rise2-cookie-consent' ); ?></p>

            <div style="display:flex;gap:20px;flex-wrap:wrap;margin-top:16px;">
                <div style="flex:1;min-width:280px;">
                    <h3 style="margin-top:0;"><?php esc_html_e( 'Export', 'rise2-cookie-consent' ); ?></h3>
                    <p class="description"><?php esc_html_e( 'Download all settings and cookies as a JSON file.', 'rise2-cookie-consent' ); ?></p>
                    <button type="button" class="button button-primary" id="r2cc-export-btn">
                        <span class="dashicons dashicons-download" style="margin-top:3px;margin-right:4px;"></span>
                        <?php esc_html_e( 'Export JSON', 'rise2-cookie-consent' ); ?>
                    </button>
                </div>
                <div style="flex:1;min-width:280px;">
                    <h3 style="margin-top:0;"><?php esc_html_e( 'Import', 'rise2-cookie-consent' ); ?></h3>
                    <p class="description"><?php esc_html_e( 'Upload a previously exported JSON file. This will overwrite current settings.', 'rise2-cookie-consent' ); ?></p>
                    <input type="file" id="r2cc-import-file" accept=".json" style="margin-bottom:8px;display:block;">
                    <button type="button" class="button" id="r2cc-import-btn" disabled>
                        <span class="dashicons dashicons-upload" style="margin-top:3px;margin-right:4px;"></span>
                        <?php esc_html_e( 'Import JSON', 'rise2-cookie-consent' ); ?>
                    </button>
                    <div id="r2cc-import-result" style="display:none;margin-top:10px;" class="r2cc-scan-stats"></div>
                </div>
            </div>
        </div>

        <!-- Shortcodes Reference -->
        <div class="r2cc-card">
            <h2 class="r2cc-card-title"><?php esc_html_e( 'Shortcodes', 'rise2-cookie-consent' ); ?></h2>
            <p class="r2cc-card-desc"><?php esc_html_e( 'Use these shortcodes to embed cookie consent elements on any page or post.', 'rise2-cookie-consent' ); ?></p>

            <table class="wp-list-table widefat fixed striped" style="margin-top:12px;">
                <thead>
                    <tr>
                        <th style="width:30%;"><?php esc_html_e( 'Shortcode', 'rise2-cookie-consent' ); ?></th>
                        <th><?php esc_html_e( 'Description', 'rise2-cookie-consent' ); ?></th>
                        <th style="width:30%;"><?php esc_html_e( 'Example', 'rise2-cookie-consent' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><code>[r2cc_settings_button]</code></td>
                        <td><?php esc_html_e( 'Renders a button that opens the cookie settings panel. Use on your privacy policy or cookie policy page.', 'rise2-cookie-consent' ); ?></td>
                        <td><code>[r2cc_settings_button text="Manage Cookies" type="link"]</code></td>
                    </tr>
                    <tr>
                        <td><code>[r2cc_cookie_table]</code></td>
                        <td><?php esc_html_e( 'Renders a full cookie declaration table with all categories. Required by some data protection authorities.', 'rise2-cookie-consent' ); ?></td>
                        <td><code>[r2cc_cookie_table category="analytics,marketing"]</code></td>
                    </tr>
                    <tr>
                        <td><code>[r2cc_consent_status]</code></td>
                        <td><?php esc_html_e( 'Shows the visitor their current consent status per category, with a button to change preferences.', 'rise2-cookie-consent' ); ?></td>
                        <td><code>[r2cc_consent_status show_reset="yes"]</code></td>
                    </tr>
                </tbody>
            </table>
        </div>

        <!-- Auto-Blocker Patterns -->
        <?php if ( get_option( 'r2cc_auto_block_scripts', 0 ) ) : ?>
        <div class="r2cc-card">
            <h2 class="r2cc-card-title"><?php esc_html_e( 'Auto-Blocker — Detected Patterns', 'rise2-cookie-consent' ); ?></h2>
            <p class="r2cc-card-desc"><?php esc_html_e( 'The auto-blocker intercepts the following services. Scripts are blocked until consent; iframes show a placeholder with an accept button.', 'rise2-cookie-consent' ); ?></p>

            <?php
            $patterns = R2CC_Auto_Blocker::get_all_patterns();
            $cat_labels = array(
                'analytics'  => __( 'Analytics', 'rise2-cookie-consent' ),
                'marketing'  => __( 'Marketing', 'rise2-cookie-consent' ),
                'functional' => __( 'Functional', 'rise2-cookie-consent' ),
            );
            ?>
            <div style="display:flex;gap:20px;flex-wrap:wrap;margin-top:12px;">
                <div style="flex:1;min-width:200px;">
                    <h4><?php esc_html_e( 'Scripts', 'rise2-cookie-consent' ); ?> (<?php echo count( $patterns['scripts'] ); ?>)</h4>
                    <ul style="font-size:12px;max-height:200px;overflow-y:auto;">
                        <?php foreach ( $patterns['scripts'] as $pattern => $cat ) : ?>
                            <li><code><?php echo esc_html( $pattern ); ?></code> → <span class="r2cc-cat-pill r2cc-cat-<?php echo esc_attr( $cat ); ?>"><?php echo esc_html( $cat_labels[ $cat ] ?? $cat ); ?></span></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <div style="flex:1;min-width:200px;">
                    <h4><?php esc_html_e( 'Iframes', 'rise2-cookie-consent' ); ?> (<?php echo count( $patterns['iframes'] ); ?>)</h4>
                    <ul style="font-size:12px;">
                        <?php foreach ( $patterns['iframes'] as $pattern => $cat ) : ?>
                            <li><code><?php echo esc_html( $pattern ); ?></code> → <span class="r2cc-cat-pill r2cc-cat-<?php echo esc_html( $cat ); ?>"><?php echo esc_html( $cat_labels[ $cat ] ?? $cat ); ?></span></li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
        </div>
        <?php endif; ?>

        <!-- Reset -->
        <div class="r2cc-card">
            <h2 class="r2cc-card-title" style="color:#dc2626;"><?php esc_html_e( 'Danger Zone', 'rise2-cookie-consent' ); ?></h2>
            <p class="description"><?php esc_html_e( 'Reset all plugin settings to factory defaults. This cannot be undone.', 'rise2-cookie-consent' ); ?></p>
            <button type="button" class="button" id="r2cc-reset-all" style="margin-top:10px;color:#dc2626;border-color:#dc2626;">
                <?php esc_html_e( 'Reset All Settings', 'rise2-cookie-consent' ); ?>
            </button>
        </div>
        <?php
    }

    /* ============================================================
       TAB: General Settings
       ============================================================ */
    private function render_tab_general() {
        ?>
        <form method="post" action="options.php">
            <?php settings_fields( 'r2cc_settings_general' ); ?>

            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'General Settings', 'rise2-cookie-consent' ); ?></h2>

                <table class="form-table r2cc-form-table">
                    <tr>
                        <th><label for="r2cc_enabled"><?php esc_html_e( 'Enable Cookie Banner', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_enabled" value="0">
                                <input type="checkbox" id="r2cc_enabled" name="r2cc_enabled" value="1" <?php checked( get_option( 'r2cc_enabled', 1 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_banner_position"><?php esc_html_e( 'Banner Position', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <select id="r2cc_banner_position" name="r2cc_banner_position" class="r2cc-select">
                                <option value="bottom-left" <?php selected( get_option( 'r2cc_banner_position' ), 'bottom-left' ); ?>><?php esc_html_e( 'Bottom Left', 'rise2-cookie-consent' ); ?></option>
                                <option value="bottom-right" <?php selected( get_option( 'r2cc_banner_position' ), 'bottom-right' ); ?>><?php esc_html_e( 'Bottom Right', 'rise2-cookie-consent' ); ?></option>
                                <option value="center-overlay" <?php selected( get_option( 'r2cc_banner_position' ), 'center-overlay' ); ?>><?php esc_html_e( 'Center Overlay (Blocking)', 'rise2-cookie-consent' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'Center overlay blocks user interaction until they make a choice.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_show_reject_all"><?php esc_html_e( 'Show "Reject All" Button', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_show_reject_all" value="0">
                                <input type="checkbox" id="r2cc_show_reject_all" name="r2cc_show_reject_all" value="1" <?php checked( get_option( 'r2cc_show_reject_all', 1 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                            <p class="description"><?php esc_html_e( 'Required by GDPR – rejecting must be as easy as accepting.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_auto_block_scripts"><?php esc_html_e( 'Auto-block Scripts & Iframes', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_auto_block_scripts" value="0">
                                <input type="checkbox" id="r2cc_auto_block_scripts" name="r2cc_auto_block_scripts" value="1" <?php checked( get_option( 'r2cc_auto_block_scripts', 0 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                            <p class="description"><?php esc_html_e( 'Automatically detects and blocks 60+ known third-party services (Google Analytics, Facebook Pixel, YouTube, Hotjar, etc.) before they load. Iframes are replaced with consent placeholders. Works with page caching.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_consent_version"><?php esc_html_e( 'Consent Version', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="text" id="r2cc_consent_version" name="r2cc_consent_version" value="<?php echo esc_attr( get_option( 'r2cc_consent_version', '1.0' ) ); ?>" class="small-text" placeholder="1.0">
                            <p class="description"><?php esc_html_e( 'Change this to force all visitors to re-consent. Useful when you update your cookie policy or add new cookie categories. E.g., change from "1.0" to "1.1".', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_consent_expiry_days"><?php esc_html_e( 'Consent Expiry (days)', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="number" id="r2cc_consent_expiry_days" name="r2cc_consent_expiry_days" value="<?php echo esc_attr( get_option( 'r2cc_consent_expiry_days', 365 ) ); ?>" min="1" max="730" class="small-text">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_privacy_policy_url"><?php esc_html_e( 'Privacy Policy URL', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="url" id="r2cc_privacy_policy_url" name="r2cc_privacy_policy_url" value="<?php echo esc_url( get_option( 'r2cc_privacy_policy_url' ) ); ?>" class="regular-text" placeholder="https://yoursite.com/privacy-policy">
                            <?php
                            $pp_page = get_option( 'wp_page_for_privacy_policy' );
                            if ( $pp_page && ! get_option( 'r2cc_privacy_policy_url' ) ) :
                                ?>
                                <p class="description">
                                    <?php printf(
                                        /* translators: %s: privacy policy page URL */
                                        esc_html__( 'Detected WordPress Privacy Policy page: %s', 'rise2-cookie-consent' ),
                                        '<a href="' . esc_url( get_permalink( $pp_page ) ) . '" target="_blank">' . esc_html( get_the_title( $pp_page ) ) . '</a>'
                                    ); ?>
                                </p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_show_powered_by"><?php esc_html_e( 'Show "Powered by"', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_show_powered_by" value="0">
                                <input type="checkbox" id="r2cc_show_powered_by" name="r2cc_show_powered_by" value="1" <?php checked( get_option( 'r2cc_show_powered_by', 1 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_scan_frequency"><?php esc_html_e( 'Automatic Cookie Scan', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <select id="r2cc_scan_frequency" name="r2cc_scan_frequency" class="r2cc-select">
                                <option value="disabled" <?php selected( get_option( 'r2cc_scan_frequency', 'r2cc_weekly' ), 'disabled' ); ?>><?php esc_html_e( 'Disabled', 'rise2-cookie-consent' ); ?></option>
                                <option value="r2cc_daily" <?php selected( get_option( 'r2cc_scan_frequency', 'r2cc_weekly' ), 'r2cc_daily' ); ?>><?php esc_html_e( 'Daily', 'rise2-cookie-consent' ); ?></option>
                                <option value="r2cc_weekly" <?php selected( get_option( 'r2cc_scan_frequency', 'r2cc_weekly' ), 'r2cc_weekly' ); ?>><?php esc_html_e( 'Weekly', 'rise2-cookie-consent' ); ?></option>
                            </select>
                            <?php
                            $next = R2CC_Cron::get_next_scan();
                            if ( $next ) :
                                ?>
                                <p class="description"><?php printf( esc_html__( 'Next scan: %s', 'rise2-cookie-consent' ), esc_html( date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $next ) ) ); ?></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_scan_email_notify"><?php esc_html_e( 'Email on new cookies', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_scan_email_notify" value="0">
                                <input type="checkbox" id="r2cc_scan_email_notify" name="r2cc_scan_email_notify" value="1" <?php checked( get_option( 'r2cc_scan_email_notify', 0 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                            <p class="description"><?php esc_html_e( 'Send an email to the admin when scheduled scans detect new cookies.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Banner Texts', 'rise2-cookie-consent' ); ?></h2>

                <table class="form-table r2cc-form-table">
                    <tr>
                        <th><label for="r2cc_banner_title"><?php esc_html_e( 'Title', 'rise2-cookie-consent' ); ?></label></th>
                        <td><input type="text" id="r2cc_banner_title" name="r2cc_banner_title" value="<?php echo esc_attr( get_option( 'r2cc_banner_title' ) ); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_banner_description"><?php esc_html_e( 'Description', 'rise2-cookie-consent' ); ?></label></th>
                        <td><textarea id="r2cc_banner_description" name="r2cc_banner_description" rows="4" class="large-text"><?php echo esc_textarea( get_option( 'r2cc_banner_description' ) ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_btn_accept_label"><?php esc_html_e( 'Accept Button', 'rise2-cookie-consent' ); ?></label></th>
                        <td><input type="text" id="r2cc_btn_accept_label" name="r2cc_btn_accept_label" value="<?php echo esc_attr( get_option( 'r2cc_btn_accept_label' ) ); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_btn_reject_label"><?php esc_html_e( 'Reject Button', 'rise2-cookie-consent' ); ?></label></th>
                        <td><input type="text" id="r2cc_btn_reject_label" name="r2cc_btn_reject_label" value="<?php echo esc_attr( get_option( 'r2cc_btn_reject_label' ) ); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_btn_settings_label"><?php esc_html_e( 'Settings Button', 'rise2-cookie-consent' ); ?></label></th>
                        <td><input type="text" id="r2cc_btn_settings_label" name="r2cc_btn_settings_label" value="<?php echo esc_attr( get_option( 'r2cc_btn_settings_label' ) ); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_btn_save_label"><?php esc_html_e( 'Save Button', 'rise2-cookie-consent' ); ?></label></th>
                        <td><input type="text" id="r2cc_btn_save_label" name="r2cc_btn_save_label" value="<?php echo esc_attr( get_option( 'r2cc_btn_save_label' ) ); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_privacy_policy_label"><?php esc_html_e( 'Privacy Link Text', 'rise2-cookie-consent' ); ?></label></th>
                        <td><input type="text" id="r2cc_privacy_policy_label" name="r2cc_privacy_policy_label" value="<?php echo esc_attr( get_option( 'r2cc_privacy_policy_label' ) ); ?>" class="regular-text"></td>
                    </tr>
                </table>
            </div>

            <?php submit_button( __( 'Save Settings', 'rise2-cookie-consent' ) ); ?>
        </form>
        <?php
    }

    /* ============================================================
       TAB: Banner & Design
       ============================================================ */
    private function render_tab_banner() {
        ?>
        <form method="post" action="options.php">
            <?php settings_fields( 'r2cc_settings_banner' ); ?>

            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Colors', 'rise2-cookie-consent' ); ?></h2>
                <p class="r2cc-card-desc"><?php esc_html_e( 'Customize the banner and button colors. Font is inherited from your theme.', 'rise2-cookie-consent' ); ?></p>

                <div class="r2cc-color-grid">
                    <?php
                    $colors = array(
                        'banner_bg_color'   => __( 'Banner Background', 'rise2-cookie-consent' ),
                        'banner_text_color'  => __( 'Banner Text', 'rise2-cookie-consent' ),
                        'btn_accept_bg'     => __( 'Accept Button BG', 'rise2-cookie-consent' ),
                        'btn_accept_text'   => __( 'Accept Button Text', 'rise2-cookie-consent' ),
                        'btn_reject_bg'     => __( 'Reject Button BG', 'rise2-cookie-consent' ),
                        'btn_reject_text'   => __( 'Reject Button Text', 'rise2-cookie-consent' ),
                        'btn_settings_bg'   => __( 'Settings Button BG', 'rise2-cookie-consent' ),
                        'btn_settings_text' => __( 'Settings Button Text', 'rise2-cookie-consent' ),
                    );

                    foreach ( $colors as $key => $label ) :
                        $value = get_option( 'r2cc_' . $key, '' );
                        ?>
                        <div class="r2cc-color-field">
                            <label><?php echo esc_html( $label ); ?></label>
                            <input type="text" name="r2cc_<?php echo esc_attr( $key ); ?>" value="<?php echo esc_attr( $value ); ?>" class="r2cc-color-picker" data-default-color="<?php echo esc_attr( $value ); ?>">
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>

            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Custom CSS', 'rise2-cookie-consent' ); ?></h2>
                <p class="r2cc-card-desc"><?php esc_html_e( 'Add custom CSS rules to further style the banner and settings panel.', 'rise2-cookie-consent' ); ?></p>

                <textarea id="r2cc_custom_css" name="r2cc_custom_css" rows="10" class="large-text code" placeholder="/* Custom CSS */"><?php echo esc_textarea( get_option( 'r2cc_custom_css', '' ) ); ?></textarea>

                <div class="r2cc-css-hints">
                    <p><strong><?php esc_html_e( 'Useful selectors:', 'rise2-cookie-consent' ); ?></strong></p>
                    <code>.r2cc-banner</code> — <?php esc_html_e( 'Main banner container', 'rise2-cookie-consent' ); ?><br>
                    <code>.r2cc-banner-inner</code> — <?php esc_html_e( 'Banner box', 'rise2-cookie-consent' ); ?><br>
                    <code>.r2cc-settings-panel</code> — <?php esc_html_e( 'Settings overlay', 'rise2-cookie-consent' ); ?><br>
                    <code>.r2cc-btn-accept</code> — <?php esc_html_e( 'Accept button', 'rise2-cookie-consent' ); ?><br>
                    <code>.r2cc-btn-reject</code> — <?php esc_html_e( 'Reject button', 'rise2-cookie-consent' ); ?><br>
                    <code>.r2cc-reopen</code> — <?php esc_html_e( 'Floating cookie icon', 'rise2-cookie-consent' ); ?>
                </div>
            </div>

            <?php submit_button( __( 'Save Design', 'rise2-cookie-consent' ) ); ?>
        </form>
        <?php
    }

    /* ============================================================
       TAB: Cookies Management
       ============================================================ */
    private function render_tab_cookies() {
        $cookies = R2CC_Cookie_Database::get_cookies();
        $categories = R2CC_Cookie_Categories::get_category_options();
        $counts = R2CC_Cookie_Database::get_cookie_counts();
        ?>
        <div class="r2cc-card">
            <div class="r2cc-card-header-row">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Manage Cookies', 'rise2-cookie-consent' ); ?></h2>
                <button type="button" class="button button-primary" id="r2cc-add-cookie">
                    <span class="dashicons dashicons-plus-alt2" style="margin-top:3px;"></span>
                    <?php esc_html_e( 'Add Cookie', 'rise2-cookie-consent' ); ?>
                </button>
            </div>

            <div class="r2cc-cookie-summary">
                <?php foreach ( $categories as $slug => $label ) :
                    $count = $counts[ $slug ] ?? 0;
                    ?>
                    <span class="r2cc-cookie-badge r2cc-badge-<?php echo esc_attr( $slug ); ?>">
                        <?php echo esc_html( $label ); ?>: <strong><?php echo intval( $count ); ?></strong>
                    </span>
                <?php endforeach; ?>
            </div>

            <!-- Add/Edit Cookie Modal -->
            <div id="r2cc-cookie-modal" class="r2cc-modal" style="display:none;">
                <div class="r2cc-modal-inner">
                    <h3 id="r2cc-modal-title"><?php esc_html_e( 'Add Cookie', 'rise2-cookie-consent' ); ?></h3>
                    <input type="hidden" id="r2cc-cookie-id" value="0">

                    <div class="r2cc-form-row">
                        <label><?php esc_html_e( 'Cookie Name', 'rise2-cookie-consent' ); ?> <span class="required">*</span></label>
                        <input type="text" id="r2cc-cookie-name" class="regular-text" placeholder="_ga, session_id, etc.">
                    </div>

                    <div class="r2cc-form-row">
                        <label><?php esc_html_e( 'Category', 'rise2-cookie-consent' ); ?></label>
                        <select id="r2cc-cookie-category" class="r2cc-select">
                            <?php foreach ( $categories as $slug => $label ) : ?>
                                <option value="<?php echo esc_attr( $slug ); ?>"><?php echo esc_html( $label ); ?></option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="r2cc-form-row">
                        <label><?php esc_html_e( 'Domain', 'rise2-cookie-consent' ); ?></label>
                        <input type="text" id="r2cc-cookie-domain" class="regular-text" placeholder="<?php echo esc_attr( parse_url( home_url(), PHP_URL_HOST ) ); ?>">
                    </div>

                    <div class="r2cc-form-row">
                        <label><?php esc_html_e( 'Duration', 'rise2-cookie-consent' ); ?></label>
                        <input type="text" id="r2cc-cookie-duration" class="regular-text" placeholder="1 year, Session, 30 days, etc.">
                    </div>

                    <div class="r2cc-form-row">
                        <label><?php esc_html_e( 'Description', 'rise2-cookie-consent' ); ?></label>
                        <textarea id="r2cc-cookie-description" rows="3" class="large-text" placeholder="<?php esc_attr_e( 'Describe what this cookie does and why it is used.', 'rise2-cookie-consent' ); ?>"></textarea>
                    </div>

                    <div class="r2cc-form-row">
                        <label>
                            <input type="checkbox" id="r2cc-cookie-regex" value="1">
                            <?php esc_html_e( 'Wildcard pattern (e.g. _ga_* matches _ga_XXXXX)', 'rise2-cookie-consent' ); ?>
                        </label>
                    </div>

                    <div class="r2cc-modal-actions">
                        <button type="button" class="button button-primary" id="r2cc-save-cookie"><?php esc_html_e( 'Save Cookie', 'rise2-cookie-consent' ); ?></button>
                        <button type="button" class="button" id="r2cc-cancel-cookie"><?php esc_html_e( 'Cancel', 'rise2-cookie-consent' ); ?></button>
                    </div>
                </div>
            </div>

            <!-- Cookies Table -->
            <table class="wp-list-table widefat fixed striped r2cc-cookies-table">
                <thead>
                    <tr>
                        <th class="column-name"><?php esc_html_e( 'Cookie Name', 'rise2-cookie-consent' ); ?></th>
                        <th class="column-category"><?php esc_html_e( 'Category', 'rise2-cookie-consent' ); ?></th>
                        <th class="column-domain"><?php esc_html_e( 'Domain', 'rise2-cookie-consent' ); ?></th>
                        <th class="column-duration"><?php esc_html_e( 'Duration', 'rise2-cookie-consent' ); ?></th>
                        <th class="column-description"><?php esc_html_e( 'Description', 'rise2-cookie-consent' ); ?></th>
                        <th class="column-actions"><?php esc_html_e( 'Actions', 'rise2-cookie-consent' ); ?></th>
                    </tr>
                </thead>
                <tbody id="r2cc-cookies-tbody">
                    <?php if ( empty( $cookies ) ) : ?>
                        <tr class="r2cc-no-cookies"><td colspan="6"><?php esc_html_e( 'No cookies found. Run a scan or add cookies manually.', 'rise2-cookie-consent' ); ?></td></tr>
                    <?php else :
                        foreach ( $cookies as $cookie ) :
                            $cat_label = R2CC_Cookie_Categories::get_category_label( $cookie->category );
                            ?>
                            <tr data-id="<?php echo intval( $cookie->id ); ?>">
                                <td>
                                    <code><?php echo esc_html( $cookie->cookie_name ); ?></code>
                                    <?php if ( $cookie->is_regex ) : ?>
                                        <span class="r2cc-tag"><?php esc_html_e( 'wildcard', 'rise2-cookie-consent' ); ?></span>
                                    <?php endif; ?>
                                </td>
                                <td><span class="r2cc-cat-pill r2cc-cat-<?php echo esc_attr( $cookie->category ); ?>"><?php echo esc_html( $cat_label ); ?></span></td>
                                <td><?php echo esc_html( $cookie->domain ?: '—' ); ?></td>
                                <td><?php echo esc_html( $cookie->duration ?: '—' ); ?></td>
                                <td class="r2cc-desc-cell"><?php echo esc_html( $cookie->description ?: '—' ); ?></td>
                                <td>
                                    <button type="button" class="button button-small r2cc-edit-cookie"
                                        data-id="<?php echo intval( $cookie->id ); ?>"
                                        data-name="<?php echo esc_attr( $cookie->cookie_name ); ?>"
                                        data-category="<?php echo esc_attr( $cookie->category ); ?>"
                                        data-domain="<?php echo esc_attr( $cookie->domain ); ?>"
                                        data-duration="<?php echo esc_attr( $cookie->duration ); ?>"
                                        data-description="<?php echo esc_attr( $cookie->description ); ?>"
                                        data-regex="<?php echo intval( $cookie->is_regex ); ?>"
                                    ><?php esc_html_e( 'Edit', 'rise2-cookie-consent' ); ?></button>
                                    <button type="button" class="button button-small button-link-delete r2cc-delete-cookie" data-id="<?php echo intval( $cookie->id ); ?>"><?php esc_html_e( 'Delete', 'rise2-cookie-consent' ); ?></button>
                                </td>
                            </tr>
                        <?php endforeach;
                    endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    /* ============================================================
       TAB: Scanner
       ============================================================ */
    private function render_tab_scanner() {
        ?>
        <div class="r2cc-card">
            <h2 class="r2cc-card-title"><?php esc_html_e( 'Cookie Scanner', 'rise2-cookie-consent' ); ?></h2>
            <p class="r2cc-card-desc"><?php esc_html_e( 'Scan your site to automatically detect cookies set by WordPress core, active plugins, and embedded third-party scripts. Detected cookies will be categorized and added to your cookie list.', 'rise2-cookie-consent' ); ?></p>

            <div class="r2cc-scanner-actions">
                <button type="button" class="button button-primary button-hero" id="r2cc-run-scan">
                    <span class="dashicons dashicons-search" style="margin-top:5px;margin-right:4px;"></span>
                    <?php esc_html_e( 'Run Cookie Scan', 'rise2-cookie-consent' ); ?>
                </button>
            </div>

            <div id="r2cc-scan-results" style="display:none;">
                <div class="r2cc-scan-stats"></div>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Cookie', 'rise2-cookie-consent' ); ?></th>
                            <th><?php esc_html_e( 'Category', 'rise2-cookie-consent' ); ?></th>
                            <th><?php esc_html_e( 'Duration', 'rise2-cookie-consent' ); ?></th>
                            <th><?php esc_html_e( 'Source', 'rise2-cookie-consent' ); ?></th>
                        </tr>
                    </thead>
                    <tbody id="r2cc-scan-tbody"></tbody>
                </table>
            </div>

            <div class="r2cc-scanner-info">
                <h3><?php esc_html_e( 'How the scanner works:', 'rise2-cookie-consent' ); ?></h3>
                <ul>
                    <li><strong><?php esc_html_e( 'Server scan:', 'rise2-cookie-consent' ); ?></strong> <?php esc_html_e( 'Detects cookies from active plugins (WooCommerce, GA, Facebook Pixel, etc.) and page source patterns.', 'rise2-cookie-consent' ); ?></li>
                    <li><strong><?php esc_html_e( 'Browser scan:', 'rise2-cookie-consent' ); ?></strong> <?php esc_html_e( 'Loads your site in a hidden frame and reads real cookies set by JavaScript. Catches cookies the server scan misses.', 'rise2-cookie-consent' ); ?></li>
                    <li><?php esc_html_e( 'Unknown cookies are imported as "Functional" — review and recategorize them in the Cookies tab.', 'rise2-cookie-consent' ); ?></li>
                    <li><?php esc_html_e( 'WPML, Elementor, Yoast, and other popular plugins', 'rise2-cookie-consent' ); ?></li>
                </ul>
            </div>
        </div>
        <?php
    }

    /* ============================================================
       TAB: Google Consent Mode v2
       ============================================================ */
    private function render_tab_gcm() {
        ?>
        <form method="post" action="options.php">
            <?php settings_fields( 'r2cc_settings_gcm' ); ?>

            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Google Consent Mode v2', 'rise2-cookie-consent' ); ?></h2>
                <p class="r2cc-card-desc"><?php esc_html_e( 'Integrates with Google Tag Manager and Google Analytics to respect user consent choices. The default consent state is output in <head> before any Google tags, and gets updated when the user makes a consent choice.', 'rise2-cookie-consent' ); ?></p>

                <table class="form-table r2cc-form-table">
                    <tr>
                        <th><label for="r2cc_gcm_enabled"><?php esc_html_e( 'Enable Google Consent Mode v2', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_gcm_enabled" value="0">
                                <input type="checkbox" id="r2cc_gcm_enabled" name="r2cc_gcm_enabled" value="1" <?php checked( get_option( 'r2cc_gcm_enabled', 0 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_gcm_wait_for_update"><?php esc_html_e( 'Wait for Update (ms)', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="number" id="r2cc_gcm_wait_for_update" name="r2cc_gcm_wait_for_update" value="<?php echo esc_attr( get_option( 'r2cc_gcm_wait_for_update', 500 ) ); ?>" min="0" max="10000" class="small-text">
                            <p class="description"><?php esc_html_e( 'How long Google tags wait for consent update before firing. 500ms is recommended.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_gcm_ads_data_redaction"><?php esc_html_e( 'Enable Ads Data Redaction', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_gcm_ads_data_redaction" value="0">
                                <input type="checkbox" id="r2cc_gcm_ads_data_redaction" name="r2cc_gcm_ads_data_redaction" value="1" <?php checked( get_option( 'r2cc_gcm_ads_data_redaction', 1 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                            <p class="description"><?php esc_html_e( 'When ad_storage is denied, redacts ad click identifiers from URLs.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_gcm_url_passthrough"><?php esc_html_e( 'Enable URL Passthrough', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <label class="r2cc-switch">
                                <input type="hidden" name="r2cc_gcm_url_passthrough" value="0">
                                <input type="checkbox" id="r2cc_gcm_url_passthrough" name="r2cc_gcm_url_passthrough" value="1" <?php checked( get_option( 'r2cc_gcm_url_passthrough', 0 ), 1 ); ?>>
                                <span class="r2cc-switch-slider"></span>
                            </label>
                            <p class="description"><?php esc_html_e( 'Passes ad click information through URLs when ad_storage is denied. Helps preserve conversion measurement.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc_gcm_region"><?php esc_html_e( 'Region Restriction', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="text" id="r2cc_gcm_region" name="r2cc_gcm_region" value="<?php echo esc_attr( get_option( 'r2cc_gcm_region', '' ) ); ?>" class="regular-text" placeholder="<?php esc_attr_e( 'e.g., AT,BE,BG,HR,CY,CZ,DK,EE,FI,FR,DE,GR,HU,IE,IT,LV,LT,LU,MT,NL,PL,PT,RO,SK,SI,ES,SE', 'rise2-cookie-consent' ); ?>">
                            <p class="description"><?php esc_html_e( 'Comma-separated ISO 3166-1 alpha-2 country codes. Leave empty to apply worldwide. Use EU country codes to restrict consent mode to EU visitors only.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                </table>
            </div>

            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Category → Google Consent Type Mapping', 'rise2-cookie-consent' ); ?></h2>
                <p class="r2cc-card-desc"><?php esc_html_e( 'This shows how your cookie categories map to Google Consent Mode v2 consent types. This mapping is automatic and cannot be changed.', 'rise2-cookie-consent' ); ?></p>

                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e( 'Cookie Category', 'rise2-cookie-consent' ); ?></th>
                            <th><?php esc_html_e( 'Google Consent Types', 'rise2-cookie-consent' ); ?></th>
                            <th><?php esc_html_e( 'Default State', 'rise2-cookie-consent' ); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <tr>
                            <td><span class="r2cc-cat-pill r2cc-cat-necessary"><?php esc_html_e( 'Strictly Necessary', 'rise2-cookie-consent' ); ?></span></td>
                            <td><code>security_storage</code></td>
                            <td><span style="color:#16a34a;font-weight:600;">✓ <?php esc_html_e( 'Always Granted', 'rise2-cookie-consent' ); ?></span></td>
                        </tr>
                        <tr>
                            <td><span class="r2cc-cat-pill r2cc-cat-functional"><?php esc_html_e( 'Functional', 'rise2-cookie-consent' ); ?></span></td>
                            <td><code>functionality_storage</code>, <code>personalization_storage</code></td>
                            <td><span style="color:#dc2626;">✗ <?php esc_html_e( 'Denied until consent', 'rise2-cookie-consent' ); ?></span></td>
                        </tr>
                        <tr>
                            <td><span class="r2cc-cat-pill r2cc-cat-analytics"><?php esc_html_e( 'Analytics', 'rise2-cookie-consent' ); ?></span></td>
                            <td><code>analytics_storage</code></td>
                            <td><span style="color:#dc2626;">✗ <?php esc_html_e( 'Denied until consent', 'rise2-cookie-consent' ); ?></span></td>
                        </tr>
                        <tr>
                            <td><span class="r2cc-cat-pill r2cc-cat-marketing"><?php esc_html_e( 'Marketing', 'rise2-cookie-consent' ); ?></span></td>
                            <td><code>ad_storage</code>, <code>ad_user_data</code>, <code>ad_personalization</code></td>
                            <td><span style="color:#dc2626;">✗ <?php esc_html_e( 'Denied until consent', 'rise2-cookie-consent' ); ?></span></td>
                        </tr>
                    </tbody>
                </table>
            </div>

            <?php submit_button( __( 'Save Settings', 'rise2-cookie-consent' ) ); ?>
        </form>
        <?php
    }

    /* ============================================================
       TAB: Privacy Policy Generator
       ============================================================ */
    private function render_tab_privacy_policy() {
        $current_lang = R2CC_I18n::get_current_language();
        ?>
        <div class="r2cc-card">
            <h2 class="r2cc-card-title"><?php esc_html_e( 'Privacy Policy Page Generator', 'rise2-cookie-consent' ); ?></h2>
            <p class="r2cc-card-desc"><?php esc_html_e( 'Generate a comprehensive GDPR-compliant privacy policy page. It will include your cookie list, data processing information, and user rights. The page is created as a draft — review and publish it when ready.', 'rise2-cookie-consent' ); ?></p>

            <div id="r2cc-pp-form">
                <table class="form-table r2cc-form-table">
                    <tr>
                        <th><label for="r2cc-pp-lang"><?php esc_html_e( 'Language', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <select id="r2cc-pp-lang" class="r2cc-select">
                                <option value="en" <?php selected( $current_lang, 'en' ); ?>><?php esc_html_e( 'English', 'rise2-cookie-consent' ); ?></option>
                                <option value="hr" <?php selected( $current_lang, 'hr' ); ?>><?php esc_html_e( 'Croatian (Hrvatski)', 'rise2-cookie-consent' ); ?></option>
                            </select>
                            <p class="description"><?php esc_html_e( 'Select the language for the privacy policy. You can generate separate pages for each language.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc-pp-company"><?php esc_html_e( 'Company / Legal Entity Name', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="text" id="r2cc-pp-company" class="regular-text" value="<?php echo esc_attr( get_option( 'r2cc_pp_company_name', get_bloginfo( 'name' ) ) ); ?>" placeholder="<?php echo esc_attr( get_bloginfo( 'name' ) ); ?>">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc-pp-address"><?php esc_html_e( 'Company Address', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="text" id="r2cc-pp-address" class="regular-text" value="<?php echo esc_attr( get_option( 'r2cc_pp_company_address', '' ) ); ?>" placeholder="<?php esc_attr_e( 'Street, City, Country', 'rise2-cookie-consent' ); ?>">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc-pp-oib"><?php esc_html_e( 'OIB / Company ID', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="text" id="r2cc-pp-oib" class="regular-text" value="<?php echo esc_attr( get_option( 'r2cc_pp_company_oib', '' ) ); ?>" placeholder="<?php esc_attr_e( 'e.g., 12345678901', 'rise2-cookie-consent' ); ?>">
                            <p class="description"><?php esc_html_e( 'Croatian OIB, EU VAT number, or equivalent company identifier.', 'rise2-cookie-consent' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc-pp-email"><?php esc_html_e( 'Contact Email', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="email" id="r2cc-pp-email" class="regular-text" value="<?php echo esc_attr( get_option( 'admin_email' ) ); ?>">
                        </td>
                    </tr>
                    <tr>
                        <th><label for="r2cc-pp-dpo"><?php esc_html_e( 'DPO Email (optional)', 'rise2-cookie-consent' ); ?></label></th>
                        <td>
                            <input type="email" id="r2cc-pp-dpo" class="regular-text" value="<?php echo esc_attr( get_option( 'r2cc_pp_dpo_email', '' ) ); ?>" placeholder="<?php esc_attr_e( 'Data Protection Officer email (if different from contact)', 'rise2-cookie-consent' ); ?>">
                        </td>
                    </tr>
                </table>

                <div class="r2cc-pp-actions" style="display:flex;gap:10px;margin-top:10px;">
                    <button type="button" class="button button-primary button-hero" id="r2cc-generate-pp">
                        <span class="dashicons dashicons-media-document" style="margin-top:5px;margin-right:4px;"></span>
                        <?php esc_html_e( 'Generate Page', 'rise2-cookie-consent' ); ?>
                    </button>
                    <button type="button" class="button button-hero" id="r2cc-preview-pp">
                        <span class="dashicons dashicons-visibility" style="margin-top:5px;margin-right:4px;"></span>
                        <?php esc_html_e( 'Preview', 'rise2-cookie-consent' ); ?>
                    </button>
                </div>

                <div id="r2cc-pp-result" style="display:none;margin-top:20px;" class="r2cc-scan-stats"></div>
            </div>
        </div>

        <div id="r2cc-pp-preview-modal" class="r2cc-modal" style="display:none;">
            <div class="r2cc-modal-inner" style="max-width:800px;max-height:80vh;overflow-y:auto;">
                <div style="display:flex;justify-content:space-between;align-items:center;margin-bottom:16px;">
                    <h3 style="margin:0;"><?php esc_html_e( 'Privacy Policy Preview', 'rise2-cookie-consent' ); ?></h3>
                    <button type="button" class="button" id="r2cc-close-pp-preview"><?php esc_html_e( 'Close', 'rise2-cookie-consent' ); ?></button>
                </div>
                <div id="r2cc-pp-preview-content" style="background:#fff;padding:20px;border:1px solid #ddd;border-radius:8px;font-size:14px;line-height:1.7;"></div>
            </div>
        </div>

        <?php
        // Show existing privacy policy page info if present
        $pp_page = get_option( 'wp_page_for_privacy_policy' );
        if ( $pp_page && get_post( $pp_page ) ) :
            ?>
            <div class="r2cc-card">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Existing Privacy Policy Page', 'rise2-cookie-consent' ); ?></h2>
                <p>
                    <strong><?php echo esc_html( get_the_title( $pp_page ) ); ?></strong>
                    (<?php echo esc_html( get_post_status( $pp_page ) ); ?>)
                    —
                    <a href="<?php echo esc_url( get_edit_post_link( $pp_page ) ); ?>"><?php esc_html_e( 'Edit', 'rise2-cookie-consent' ); ?></a>
                    |
                    <a href="<?php echo esc_url( get_permalink( $pp_page ) ); ?>" target="_blank"><?php esc_html_e( 'View', 'rise2-cookie-consent' ); ?></a>
                </p>
            </div>
        <?php endif; ?>
        <?php
    }

    /* ============================================================
       TAB: Consent Logs
       ============================================================ */
    private function render_tab_consent() {
        $page = isset( $_GET['log_page'] ) ? max( 1, intval( $_GET['log_page'] ) ) : 1;
        $data = R2CC_Consent_Logger::get_logs( $page, 30 );
        $stats = R2CC_Consent_Logger::get_stats();
        ?>
        <div class="r2cc-card">
            <div class="r2cc-card-header-row">
                <h2 class="r2cc-card-title"><?php esc_html_e( 'Consent Log', 'rise2-cookie-consent' ); ?></h2>
                <a href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=r2cc-settings&r2cc_export_csv=1' ), 'r2cc_export' ); ?>" class="button">
                    <span class="dashicons dashicons-download" style="margin-top:3px;"></span>
                    <?php esc_html_e( 'Export CSV', 'rise2-cookie-consent' ); ?>
                </a>
            </div>

            <div class="r2cc-stats-grid">
                <div class="r2cc-stat-card">
                    <div class="r2cc-stat-number"><?php echo intval( $stats['total'] ); ?></div>
                    <div class="r2cc-stat-label"><?php esc_html_e( 'Total Consents', 'rise2-cookie-consent' ); ?></div>
                </div>
                <div class="r2cc-stat-card">
                    <div class="r2cc-stat-number"><?php echo intval( $stats['last_30_days'] ); ?></div>
                    <div class="r2cc-stat-label"><?php esc_html_e( 'Last 30 Days', 'rise2-cookie-consent' ); ?></div>
                </div>
                <?php foreach ( $stats['by_type'] as $type ) : ?>
                    <div class="r2cc-stat-card">
                        <div class="r2cc-stat-number"><?php echo intval( $type->count ); ?></div>
                        <div class="r2cc-stat-label"><?php echo esc_html( ucwords( str_replace( '_', ' ', $type->consent_type ) ) ); ?></div>
                    </div>
                <?php endforeach; ?>
            </div>

            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e( 'Consent UID', 'rise2-cookie-consent' ); ?></th>
                        <th><?php esc_html_e( 'Type', 'rise2-cookie-consent' ); ?></th>
                        <th><?php esc_html_e( 'Categories', 'rise2-cookie-consent' ); ?></th>
                        <th><?php esc_html_e( 'IP (anonymized)', 'rise2-cookie-consent' ); ?></th>
                        <th><?php esc_html_e( 'Date', 'rise2-cookie-consent' ); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( empty( $data['logs'] ) ) : ?>
                        <tr><td colspan="5"><?php esc_html_e( 'No consent records yet.', 'rise2-cookie-consent' ); ?></td></tr>
                    <?php else :
                        foreach ( $data['logs'] as $log ) :
                            $cats = json_decode( $log->categories, true );
                            ?>
                            <tr>
                                <td><code style="font-size:11px;"><?php echo esc_html( substr( $log->consent_uid, 0, 18 ) . '...' ); ?></code></td>
                                <td><span class="r2cc-type-badge r2cc-type-<?php echo esc_attr( $log->consent_type ); ?>"><?php echo esc_html( ucwords( str_replace( '_', ' ', $log->consent_type ) ) ); ?></span></td>
                                <td><?php echo is_array( $cats ) ? esc_html( implode( ', ', $cats ) ) : '—'; ?></td>
                                <td><?php echo esc_html( $log->ip_address ); ?></td>
                                <td><?php echo esc_html( $log->created_at ); ?></td>
                            </tr>
                        <?php endforeach;
                    endif; ?>
                </tbody>
            </table>

            <?php if ( $data['pages'] > 1 ) : ?>
                <div class="r2cc-pagination">
                    <?php for ( $i = 1; $i <= $data['pages']; $i++ ) :
                        $url = add_query_arg( array( 'page' => 'r2cc-settings', 'tab' => 'consent', 'log_page' => $i ), admin_url( 'admin.php' ) );
                        ?>
                        <a href="<?php echo esc_url( $url ); ?>" class="button <?php echo $i === $page ? 'button-primary' : ''; ?>"><?php echo intval( $i ); ?></a>
                    <?php endfor; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}

new R2CC_Admin();
