/**
 * RISE2 Cookie Consent – Admin Scripts
 */
(function ($) {
    'use strict';

    if (typeof r2ccAdmin === 'undefined') return;

    const A = r2ccAdmin;

    /* ============================================================
       COLOR PICKERS
       ============================================================ */
    $(document).ready(function () {
        $('.r2cc-color-picker').wpColorPicker();
    });

    /* ============================================================
       COOKIE MODAL (Add / Edit)
       ============================================================ */
    const modal = $('#r2cc-cookie-modal');

    function openModal(data) {
        data = data || {};
        $('#r2cc-modal-title').text(data.id ? A.strings.edit || 'Edit Cookie' : 'Add Cookie');
        $('#r2cc-cookie-id').val(data.id || 0);
        $('#r2cc-cookie-name').val(data.name || '');
        $('#r2cc-cookie-category').val(data.category || 'necessary');
        $('#r2cc-cookie-domain').val(data.domain || '');
        $('#r2cc-cookie-duration').val(data.duration || '');
        $('#r2cc-cookie-description').val(data.description || '');
        $('#r2cc-cookie-regex').prop('checked', parseInt(data.regex) === 1);
        modal.show();
        $('#r2cc-cookie-name').focus();
    }

    function closeModal() {
        modal.hide();
    }

    // Add Cookie button
    $(document).on('click', '#r2cc-add-cookie', function () {
        openModal();
    });

    // Edit Cookie button
    $(document).on('click', '.r2cc-edit-cookie', function () {
        const btn = $(this);
        openModal({
            id: btn.data('id'),
            name: btn.data('name'),
            category: btn.data('category'),
            domain: btn.data('domain'),
            duration: btn.data('duration'),
            description: btn.data('description'),
            regex: btn.data('regex'),
        });
    });

    // Cancel
    $(document).on('click', '#r2cc-cancel-cookie', closeModal);

    // Close on overlay click
    modal.on('click', function (e) {
        if ($(e.target).is(modal)) closeModal();
    });

    // Save Cookie
    $(document).on('click', '#r2cc-save-cookie', function () {
        const name = $('#r2cc-cookie-name').val().trim();
        if (!name) {
            alert('Cookie name is required.');
            return;
        }

        const btn = $(this);
        btn.prop('disabled', true).text('Saving...');

        $.post(A.ajaxUrl, {
            action: 'r2cc_save_cookie',
            nonce: A.nonce,
            cookie_id: $('#r2cc-cookie-id').val(),
            cookie_name: name,
            category: $('#r2cc-cookie-category').val(),
            domain: $('#r2cc-cookie-domain').val(),
            duration: $('#r2cc-cookie-duration').val(),
            description: $('#r2cc-cookie-description').val(),
            is_regex: $('#r2cc-cookie-regex').is(':checked') ? 1 : 0,
        })
            .done(function (res) {
                if (res.success) {
                    location.reload();
                } else {
                    alert('Error saving cookie.');
                }
            })
            .fail(function () {
                alert('Request failed.');
            })
            .always(function () {
                btn.prop('disabled', false).text('Save Cookie');
            });
    });

    /* ============================================================
       DELETE COOKIE
       ============================================================ */
    $(document).on('click', '.r2cc-delete-cookie', function () {
        if (!confirm(A.strings.confirmDelete)) return;

        const id = $(this).data('id');
        const row = $(this).closest('tr');

        $.post(A.ajaxUrl, {
            action: 'r2cc_delete_cookie',
            nonce: A.nonce,
            cookie_id: id,
        }).done(function (res) {
            if (res.success) {
                row.fadeOut(300, function () {
                    $(this).remove();
                });
            }
        });
    });

    /* ============================================================
       COOKIE SCANNER (hybrid: server-side + browser-based)
       ============================================================ */
    $(document).on('click', '#r2cc-run-scan', function () {
        const btn = $(this);
        btn.prop('disabled', true).html('<span class="r2cc-spinner"></span> ' + A.strings.scanning);

        const results = $('#r2cc-scan-results');
        const tbody = $('#r2cc-scan-tbody');

        // Phase 1: Server-side scan (plugin detection + HTML patterns)
        $.post(A.ajaxUrl, {
            action: 'r2cc_scan_cookies',
            nonce: A.nonce,
        })
            .done(function (res) {
                if (!res.success) return;

                const serverData = res.data;
                results.show();

                // Show server scan results immediately
                results.find('.r2cc-scan-stats').html(
                    `<strong>${serverData.detected}</strong> cookies detected (server scan), <strong>${serverData.imported}</strong> new. <em>Browser scan running...</em>`
                );

                showScanTable(tbody, serverData.cookies, 'Server');

                // Phase 2: Browser-based scan (real cookies from document.cookie)
                runBrowserScan(function (browserResult) {
                    // Update stats with combined totals
                    const totalDetected = serverData.detected + (browserResult ? browserResult.detected : 0);
                    const totalImported = serverData.imported + (browserResult ? browserResult.imported : 0);
                    let statsHtml = `<strong>${totalDetected}</strong> cookies detected, <strong>${totalImported}</strong> new imported.`;

                    if (browserResult && browserResult.new_cookies && browserResult.new_cookies.length) {
                        statsHtml += `<br><span style="color:#d97706;">⚠ ${browserResult.new_cookies.length} unknown cookie(s) found by browser scan — please review and categorize in the Cookies tab.</span>`;
                    }

                    results.find('.r2cc-scan-stats').html(statsHtml);

                    // Re-render full table
                    if (browserResult && browserResult.detected > 0) {
                        // Reload the page to show updated cookie list
                        results.find('.r2cc-scan-stats').append(' <a href="' + window.location.href + '">Refresh to see all cookies</a>');
                    }

                    btn.prop('disabled', false).html('<span class="dashicons dashicons-search" style="margin-top:5px;margin-right:4px;"></span> Run Cookie Scan');
                });
            })
            .fail(function () {
                alert('Scan failed. Please try again.');
                btn.prop('disabled', false).html('<span class="dashicons dashicons-search" style="margin-top:5px;margin-right:4px;"></span> Run Cookie Scan');
            });
    });

    /**
     * Show scan results in the table.
     */
    function showScanTable(tbody, cookies, source) {
        tbody.empty();
        if (cookies && cookies.length) {
            cookies.forEach(function (c) {
                tbody.append(`<tr>
                    <td><code>${escHtml(c.cookie_name)}</code></td>
                    <td>${escHtml(c.category)}</td>
                    <td>${escHtml(c.duration || '—')}</td>
                    <td>${escHtml(source)}</td>
                </tr>`);
            });
        } else {
            tbody.append('<tr><td colspan="4">No cookies detected.</td></tr>');
        }
    }

    /**
     * Browser-based scan: loads frontend in a hidden iframe,
     * reads real cookies, and sends them to the server for matching.
     */
    function runBrowserScan(callback) {
        // Create hidden iframe to load the frontend
        const iframe = document.createElement('iframe');
        iframe.style.cssText = 'position:absolute;width:1px;height:1px;opacity:0;pointer-events:none;';
        iframe.src = A.siteUrl + '?r2cc_scan_mode=1'; // query param to avoid caching
        document.body.appendChild(iframe);

        // Wait for page to load + scripts to execute
        const onReady = function () {
            // Give async scripts 2s to set cookies
            setTimeout(function () {
                // Read all cookies from this domain
                const cookieStr = document.cookie;
                const cookieNames = [];

                if (cookieStr) {
                    cookieStr.split(';').forEach(function (c) {
                        const name = c.trim().split('=')[0];
                        if (name && cookieNames.indexOf(name) === -1) {
                            cookieNames.push(name);
                        }
                    });
                }

                // Remove iframe
                document.body.removeChild(iframe);

                if (cookieNames.length === 0) {
                    callback(null);
                    return;
                }

                // Send to server for matching
                $.post(A.ajaxUrl, {
                    action: 'r2cc_scan_browser_cookies',
                    nonce: A.nonce,
                    cookies: cookieNames,
                })
                    .done(function (res) {
                        callback(res.success ? res.data : null);
                    })
                    .fail(function () {
                        callback(null);
                    });
            }, 2000);
        };

        iframe.addEventListener('load', onReady);

        // Fallback timeout in case iframe fails to load
        setTimeout(function () {
            if (document.body.contains(iframe)) {
                onReady();
            }
        }, 8000);
    }

    /* ============================================================
       PRIVACY POLICY GENERATOR
       ============================================================ */
    function getPPFormData() {
        return {
            nonce: A.nonce,
            lang: $('#r2cc-pp-lang').val(),
            company_name: $('#r2cc-pp-company').val(),
            company_address: $('#r2cc-pp-address').val(),
            company_oib: $('#r2cc-pp-oib').val(),
            contact_email: $('#r2cc-pp-email').val(),
            dpo_email: $('#r2cc-pp-dpo').val(),
        };
    }

    $(document).on('click', '#r2cc-generate-pp', function () {
        const btn = $(this);
        btn.prop('disabled', true).html('<span class="r2cc-spinner"></span> Generating...');

        const data = getPPFormData();
        data.action = 'r2cc_generate_privacy_policy';

        $.post(A.ajaxUrl, data)
            .done(function (res) {
                if (res.success) {
                    const d = res.data;
                    const result = $('#r2cc-pp-result');
                    result.show().html(
                        `<strong>✅ ${escHtml(d.message)}</strong><br>` +
                        `<a href="${escHtml(d.edit_url)}" class="button" style="margin-top:8px;">Edit Page</a> ` +
                        `<a href="${escHtml(d.view_url)}" class="button" target="_blank" style="margin-top:8px;">View Page</a>`
                    );
                } else {
                    alert('Failed to generate privacy policy page.');
                }
            })
            .fail(function () {
                alert('Request failed.');
            })
            .always(function () {
                btn.prop('disabled', false).html('<span class="dashicons dashicons-media-document" style="margin-top:5px;margin-right:4px;"></span> Generate Page');
            });
    });

    $(document).on('click', '#r2cc-preview-pp', function () {
        const btn = $(this);
        btn.prop('disabled', true).html('<span class="r2cc-spinner"></span> Loading...');

        const data = getPPFormData();
        data.action = 'r2cc_preview_privacy_policy';

        $.post(A.ajaxUrl, data)
            .done(function (res) {
                if (res.success) {
                    $('#r2cc-pp-preview-content').html(res.data.html);
                    $('#r2cc-pp-preview-modal').show();
                }
            })
            .fail(function () {
                alert('Preview failed.');
            })
            .always(function () {
                btn.prop('disabled', false).html('<span class="dashicons dashicons-visibility" style="margin-top:5px;margin-right:4px;"></span> Preview');
            });
    });

    $(document).on('click', '#r2cc-close-pp-preview', function () {
        $('#r2cc-pp-preview-modal').hide();
    });

    $('#r2cc-pp-preview-modal').on('click', function (e) {
        if ($(e.target).is('#r2cc-pp-preview-modal')) {
            $(this).hide();
        }
    });

    /* ============================================================
       KEYBOARD: Escape closes modal
       ============================================================ */
    $(document).on('keydown', function (e) {
        if (e.key === 'Escape' && modal.is(':visible')) {
            closeModal();
        }
    });

    /* ============================================================
       IMPORT / EXPORT
       ============================================================ */
    // Export
    $(document).on('click', '#r2cc-export-btn', function () {
        const btn = $(this);
        btn.prop('disabled', true).html('<span class="r2cc-spinner"></span> Exporting...');

        $.post(A.ajaxUrl, {
            action: 'r2cc_export_settings',
            nonce: A.nonce,
        })
            .done(function (res) {
                if (res.success) {
                    const json = JSON.stringify(res.data, null, 2);
                    const blob = new Blob([json], { type: 'application/json' });
                    const url = URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = 'r2cc-settings-' + new Date().toISOString().slice(0, 10) + '.json';
                    a.click();
                    URL.revokeObjectURL(url);
                }
            })
            .always(function () {
                btn.prop('disabled', false).html('<span class="dashicons dashicons-download" style="margin-top:3px;margin-right:4px;"></span> Export JSON');
            });
    });

    // Import - enable button when file selected
    $(document).on('change', '#r2cc-import-file', function () {
        $('#r2cc-import-btn').prop('disabled', !this.files.length);
    });

    // Import
    $(document).on('click', '#r2cc-import-btn', function () {
        const fileInput = document.getElementById('r2cc-import-file');
        if (!fileInput.files.length) return;

        if (!confirm('This will overwrite your current settings. Continue?')) return;

        const btn = $(this);
        btn.prop('disabled', true).html('<span class="r2cc-spinner"></span> Importing...');

        const reader = new FileReader();
        reader.onload = function (e) {
            try {
                JSON.parse(e.target.result); // Validate JSON
            } catch {
                alert('Invalid JSON file.');
                btn.prop('disabled', false).html('<span class="dashicons dashicons-upload" style="margin-top:3px;margin-right:4px;"></span> Import JSON');
                return;
            }

            $.post(A.ajaxUrl, {
                action: 'r2cc_import_settings',
                nonce: A.nonce,
                import_data: e.target.result,
            })
                .done(function (res) {
                    const result = $('#r2cc-import-result');
                    if (res.success) {
                        result.show().html('<strong>✅ ' + escHtml(res.data.message) + '</strong>');
                        setTimeout(() => location.reload(), 1500);
                    } else {
                        result.show().html('<strong>❌ ' + escHtml(res.data) + '</strong>');
                    }
                })
                .fail(function () {
                    alert('Import failed.');
                })
                .always(function () {
                    btn.prop('disabled', false).html('<span class="dashicons dashicons-upload" style="margin-top:3px;margin-right:4px;"></span> Import JSON');
                });
        };
        reader.readAsText(fileInput.files[0]);
    });

    /* ============================================================
       RESET ALL
       ============================================================ */
    $(document).on('click', '#r2cc-reset-all', function () {
        if (!confirm('Are you sure you want to reset ALL settings to defaults? This cannot be undone.')) return;
        if (!confirm('Really sure? All settings, cookies, and consent logs will be reset.')) return;

        const btn = $(this);
        btn.prop('disabled', true).text('Resetting...');

        $.post(A.ajaxUrl, {
            action: 'r2cc_reset_settings',
            nonce: A.nonce,
        }).done(function (res) {
            if (res.success) {
                alert('Settings reset. Page will reload.');
                location.reload();
            }
        }).always(function () {
            btn.prop('disabled', false).text('Reset All Settings');
        });
    });

    /* ============================================================
       DASHBOARD CHART (Chart.js)
       ============================================================ */
    const chartCanvas = document.getElementById('r2cc-consent-chart');
    if (chartCanvas) {
        // Load Chart.js dynamically
        const script = document.createElement('script');
        script.src = 'https://cdn.jsdelivr.net/npm/chart.js@4/dist/chart.umd.min.js';
        script.onload = function () {
            loadDashboardChart(30);
        };
        document.head.appendChild(script);
    }

    function loadDashboardChart(days) {
        $.post(A.ajaxUrl, {
            action: 'r2cc_get_analytics',
            nonce: A.nonce,
            days: days,
        }).done(function (res) {
            if (!res.success || !res.data.daily) return;

            const daily = res.data.daily;
            const labels = daily.map(d => {
                const dt = new Date(d.date);
                return dt.toLocaleDateString(undefined, { month: 'short', day: 'numeric' });
            });

            const ctx = document.getElementById('r2cc-consent-chart');
            if (!ctx) return;

            // Destroy existing chart if any
            if (window.r2ccChart) window.r2ccChart.destroy();

            window.r2ccChart = new Chart(ctx, {
                type: 'line',
                data: {
                    labels: labels,
                    datasets: [
                        {
                            label: 'Accept All',
                            data: daily.map(d => d.accept_all),
                            borderColor: '#16a34a',
                            backgroundColor: 'rgba(22,163,74,0.08)',
                            fill: true,
                            tension: 0.3,
                            borderWidth: 2,
                            pointRadius: 1,
                        },
                        {
                            label: 'Reject All',
                            data: daily.map(d => d.reject_all),
                            borderColor: '#dc2626',
                            backgroundColor: 'rgba(220,38,38,0.08)',
                            fill: true,
                            tension: 0.3,
                            borderWidth: 2,
                            pointRadius: 1,
                        },
                        {
                            label: 'Custom',
                            data: daily.map(d => d.custom),
                            borderColor: '#2563eb',
                            backgroundColor: 'rgba(37,99,235,0.08)',
                            fill: true,
                            tension: 0.3,
                            borderWidth: 2,
                            pointRadius: 1,
                        },
                    ],
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    interaction: { mode: 'index', intersect: false },
                    plugins: {
                        legend: { position: 'top', labels: { usePointStyle: true, padding: 16 } },
                    },
                    scales: {
                        y: { beginAtZero: true, ticks: { precision: 0 } },
                        x: { ticks: { maxTicksLimit: 10 } },
                    },
                },
            });
        });
    }

    // Chart range change
    $(document).on('change', '#r2cc-chart-range', function () {
        loadDashboardChart(parseInt($(this).val(), 10));
    });

    /* ============================================================
       HELPER
       ============================================================ */
    function escHtml(str) {
        if (!str) return '';
        const div = document.createElement('div');
        div.textContent = str;
        return div.innerHTML;
    }

})(jQuery);
