/**
 * RISE2 Cookie Consent – Frontend Engine
 * Standalone, no dependencies (vanilla JS)
 */
(function () {
    'use strict';

    if (typeof r2ccConfig === 'undefined') return;

    const C = r2ccConfig;
    const COOKIE_NAME = 'r2cc_consent';
    const UID_COOKIE = 'r2cc_consent_uid';
    const EXPIRY_DAYS = parseInt(C.consentExpiry, 10) || 365;
    const CONSENT_VERSION = C.consentVersion || '1.0';

    /* ============================================================
       COOKIE HELPERS
       ============================================================ */
    const Cookies = {
        set(name, value, days) {
            const d = new Date();
            d.setTime(d.getTime() + days * 86400000);
            const secure = location.protocol === 'https:' ? ';Secure' : '';
            document.cookie = `${name}=${encodeURIComponent(value)};expires=${d.toUTCString()};path=/;SameSite=Lax${secure}`;
        },
        get(name) {
            const match = document.cookie.match(new RegExp('(^| )' + name + '=([^;]+)'));
            return match ? decodeURIComponent(match[2]) : null;
        },
        remove(name) {
            document.cookie = `${name}=;expires=Thu, 01 Jan 1970 00:00:00 GMT;path=/;SameSite=Lax`;
        }
    };

    /* ============================================================
       CONSENT STATE
       ============================================================ */
    const State = {
        getConsent() {
            const raw = Cookies.get(COOKIE_NAME);
            if (!raw) return null;
            try { return JSON.parse(raw); } catch { return null; }
        },

        setConsent(categories, type) {
            // Check if consent actually changed (skip duplicate logging)
            const existing = this.getConsent();
            const changed = !existing
                || existing.version !== CONSENT_VERSION
                || existing.type !== type
                || JSON.stringify([...existing.categories].sort()) !== JSON.stringify([...categories].sort());

            const data = {
                categories,
                type,
                timestamp: new Date().toISOString(),
                version: CONSENT_VERSION
            };
            Cookies.set(COOKIE_NAME, JSON.stringify(data), EXPIRY_DAYS);

            // Ensure UID exists
            if (!Cookies.get(UID_COOKIE)) {
                Cookies.set(UID_COOKIE, this.generateUID(), EXPIRY_DAYS);
            }

            // Only report to server if consent actually changed
            if (changed) {
                this.reportConsent(categories, type);
            }

            // Fire event
            window.dispatchEvent(new CustomEvent('r2cc:consent', { detail: data }));

            return data;
        },

        hasConsent() {
            const consent = this.getConsent();
            if (!consent) return false;

            // Check version — if policy version changed, invalidate old consent
            if (consent.version !== CONSENT_VERSION) {
                return false;
            }

            return true;
        },

        isCategoryAllowed(cat) {
            if (cat === 'necessary') return true;
            const consent = this.getConsent();
            if (!consent || consent.version !== CONSENT_VERSION) return false;
            return consent.categories && consent.categories.includes(cat);
        },

        reportConsent(categories, type) {
            if (!C.ajaxUrl) return;
            const fd = new FormData();
            fd.append('action', 'r2cc_save_consent');
            fd.append('nonce', C.nonce);
            fd.append('consent_type', type);
            categories.forEach(cat => fd.append('categories[]', cat));
            fetch(C.ajaxUrl, { method: 'POST', body: fd, credentials: 'same-origin' }).catch(() => {});
        },

        generateUID() {
            return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, c => {
                const r = Math.random() * 16 | 0;
                return (c === 'x' ? r : (r & 0x3 | 0x8)).toString(16);
            });
        }
    };

    /* ============================================================
       SCRIPT BLOCKING (optional)
       ============================================================ */
    const ScriptBlocker = {
        init() {
            if (!parseInt(C.autoBlockScripts, 10)) return;

            // Observe DOM for dynamically added scripts
            const observer = new MutationObserver(mutations => {
                mutations.forEach(m => {
                    m.addedNodes.forEach(node => {
                        if (node.tagName === 'SCRIPT' && node.hasAttribute('data-r2cc-category')) {
                            this.handleScript(node);
                        }
                    });
                });
            });
            observer.observe(document.documentElement, { childList: true, subtree: true });

            // Handle existing scripts with data-r2cc-category
            document.querySelectorAll('script[data-r2cc-category]').forEach(s => this.handleScript(s));
        },

        handleScript(script) {
            const cat = script.getAttribute('data-r2cc-category');
            if (!cat || cat === 'necessary') return;

            if (!State.isCategoryAllowed(cat)) {
                // Block the script by changing its type
                script.type = 'text/plain';
                script.setAttribute('data-r2cc-blocked', 'true');
            }
        },

        unblockCategory(cat) {
            document.querySelectorAll(`script[data-r2cc-category="${cat}"][data-r2cc-blocked="true"]`).forEach(script => {
                const newScript = document.createElement('script');
                for (const attr of script.attributes) {
                    if (attr.name !== 'type' && attr.name !== 'data-r2cc-blocked') {
                        newScript.setAttribute(attr.name, attr.value);
                    }
                }
                newScript.textContent = script.textContent;
                script.parentNode.replaceChild(newScript, script);
            });
        },

        unblockAll(categories) {
            categories.forEach(cat => this.unblockCategory(cat));
        }
    };

    /* ============================================================
       GOOGLE CONSENT MODE v2
       ============================================================ */
    const GCM = {
        enabled: false,
        categoryMap: {},

        init() {
            if (!C.gcm || !C.gcm.enabled) return;
            this.enabled = true;
            this.categoryMap = C.gcm.categoryMap || {};
        },

        /**
         * Update Google Consent Mode based on accepted categories
         */
        updateConsent(categories) {
            if (!this.enabled) return;
            if (typeof gtag !== 'function') return;

            const consentState = {};

            // Start with all denied
            const allTypes = [
                'ad_storage', 'ad_user_data', 'ad_personalization',
                'analytics_storage', 'functionality_storage',
                'personalization_storage', 'security_storage'
            ];
            allTypes.forEach(t => { consentState[t] = 'denied'; });

            // Security storage is always granted
            consentState['security_storage'] = 'granted';

            // Grant types based on consented categories
            categories.forEach(cat => {
                const mapped = this.categoryMap[cat];
                if (mapped) {
                    mapped.forEach(type => {
                        consentState[type] = 'granted';
                    });
                }
            });

            gtag('consent', 'update', consentState);

            // Fire dataLayer event for GTM
            window.dataLayer = window.dataLayer || [];
            window.dataLayer.push({
                event: 'r2cc_consent_update',
                r2cc_categories: categories,
                r2cc_consent_state: consentState
            });
        }
    };

    /* ============================================================
       UI – Banner & Settings Panel
       ============================================================ */
    const UI = {
        banner: null,
        settingsPanel: null,
        reopenBtn: null,

        init() {
            this.banner = document.getElementById('r2cc-banner');
            this.settingsPanel = document.getElementById('r2cc-settings-panel');
            this.reopenBtn = document.getElementById('r2cc-reopen');

            if (!this.banner) return;

            this.populateBanner();
            this.populateSettings();
            this.bindEvents();
            this.initIframePlaceholders();

            // Show banner or reopen button
            if (!State.hasConsent()) {
                this.showBanner();
            } else {
                this.showReopen();
                // Unblock previously consented scripts
                const consent = State.getConsent();
                if (consent && consent.categories) {
                    ScriptBlocker.unblockAll(consent.categories);
                }
            }
        },

        populateBanner() {
            const t = C.texts;
            this.banner.querySelector('.r2cc-banner-title').textContent = t.title;
            this.banner.querySelector('.r2cc-banner-description').textContent = t.description;
            this.banner.querySelector('.r2cc-btn-accept').textContent = t.acceptAll;
            this.banner.querySelector('.r2cc-btn-settings').textContent = t.settings;

            const rejectBtn = this.banner.querySelector('.r2cc-btn-reject');
            if (parseInt(C.showRejectAll, 10)) {
                rejectBtn.textContent = t.rejectAll;
                rejectBtn.style.display = '';
            }

            // Privacy link
            if (t.privacyUrl) {
                const wrap = this.banner.querySelector('.r2cc-privacy-link-wrap');
                wrap.innerHTML = `<a href="${this.escHtml(t.privacyUrl)}" target="_blank" rel="noopener">${this.escHtml(t.privacyLabel || 'Privacy Policy')}</a>`;
            }
        },

        populateSettings() {
            const container = this.settingsPanel.querySelector('.r2cc-categories-list');
            if (!container) return;

            const t = C.texts;
            const st = C.settingsTexts || {};
            const cats = C.categories;
            const cookiesByCat = C.cookiesByCat || {};

            // Footer buttons
            const footer = this.settingsPanel.querySelector('.r2cc-settings-footer');
            footer.querySelector('.r2cc-btn-accept').textContent = t.acceptAll;
            footer.querySelector('.r2cc-btn-reject').textContent = t.rejectAll;
            footer.querySelector('.r2cc-btn-save').textContent = t.save;

            // Translated settings panel title & description (if available)
            const stTitle = this.settingsPanel.querySelector('.r2cc-settings-title');
            if (stTitle && st.settingsTitle) stTitle.textContent = st.settingsTitle;
            const stDesc = this.settingsPanel.querySelector('.r2cc-settings-desc');
            if (stDesc && st.settingsDesc) stDesc.textContent = st.settingsDesc;

            const alwaysActiveLabel = st.alwaysActive || 'Always Active';
            const noCookiesLabel = st.noCookies || 'No cookies registered in this category yet.';

            // Build category accordions
            let html = '';
            const iconSvgs = {
                shield: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none"><path d="M12 2l8 4v6c0 5.5-3.8 10-8 12-4.2-2-8-6.5-8-12V6l8-4z" stroke="currentColor" stroke-width="2" fill="currentColor" fill-opacity="0.15"/></svg>',
                settings: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none"><path d="M12 15a3 3 0 100-6 3 3 0 000 6z" stroke="currentColor" stroke-width="2"/><path d="M19.4 15a1.65 1.65 0 00.33 1.82l.06.06a2 2 0 01-2.83 2.83l-.06-.06a1.65 1.65 0 00-1.82-.33 1.65 1.65 0 00-1 1.51V21a2 2 0 01-4 0v-.09a1.65 1.65 0 00-1.08-1.51 1.65 1.65 0 00-1.82.33l-.06.06a2 2 0 01-2.83-2.83l.06-.06a1.65 1.65 0 00.33-1.82 1.65 1.65 0 00-1.51-1H3a2 2 0 010-4h.09a1.65 1.65 0 001.51-1.08 1.65 1.65 0 00-.33-1.82l-.06-.06a2 2 0 012.83-2.83l.06.06a1.65 1.65 0 001.82.33H9a1.65 1.65 0 001-1.51V3a2 2 0 014 0v.09a1.65 1.65 0 001 1.51 1.65 1.65 0 001.82-.33l.06-.06a2 2 0 012.83 2.83l-.06.06a1.65 1.65 0 00-.33 1.82V9c.26.6.77 1.02 1.37 1.08H21a2 2 0 010 4h-.09a1.65 1.65 0 00-1.51 1z" stroke="currentColor" stroke-width="2"/></svg>',
                chart: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none"><path d="M18 20V10M12 20V4M6 20v-6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
                megaphone: '<svg width="16" height="16" viewBox="0 0 24 24" fill="none"><path d="M21 15V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10m18 0l-6-3m6 3l-6 3m-12-3l6-3m-6 3l6 3" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>',
            };

            for (const [slug, cat] of Object.entries(cats)) {
                const cookies = cookiesByCat[slug] || [];
                const isRequired = cat.is_required;
                const existingConsent = State.getConsent();
                let checked = isRequired || cat.default_on;

                // If user has existing consent, use that
                if (existingConsent && existingConsent.categories) {
                    checked = isRequired || existingConsent.categories.includes(slug);
                }

                const icon = iconSvgs[cat.icon] || iconSvgs.settings;

                html += `
                <div class="r2cc-category-item" data-category="${slug}">
                    <div class="r2cc-category-header">
                        <div class="r2cc-category-info">
                            <div class="r2cc-category-icon r2cc-category-icon-${slug}">${icon}</div>
                            <div>
                                <span class="r2cc-category-label">${this.escHtml(cat.label)}</span>
                                ${cookies.length ? `<span class="r2cc-category-count">(${cookies.length} cookie${cookies.length !== 1 ? 's' : ''})</span>` : ''}
                            </div>
                        </div>
                        ${isRequired
                            ? `<span class="r2cc-category-badge">${this.escHtml(alwaysActiveLabel)}</span>`
                            : `<label class="r2cc-toggle" onclick="event.stopPropagation()">
                                <input type="checkbox" data-cat="${slug}" ${checked ? 'checked' : ''}>
                                <span class="r2cc-toggle-slider"></span>
                              </label>`
                        }
                        <div class="r2cc-category-expand">
                            <svg width="14" height="14" viewBox="0 0 24 24" fill="none"><path d="M6 9l6 6 6-6" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>
                        </div>
                    </div>
                    <div class="r2cc-category-details">
                        <p class="r2cc-category-description">${this.escHtml(cat.description)}</p>
                        ${cookies.length ? this.buildCookieTable(cookies) : `<p style="font-size:12px;opacity:0.5;">${this.escHtml(noCookiesLabel)}</p>`}
                    </div>
                </div>`;
            }

            container.innerHTML = html;
        },

        buildCookieTable(cookies) {
            const st = C.settingsTexts || {};
            let rows = '';
            cookies.forEach(c => {
                rows += `<tr>
                    <td><span class="r2cc-cookie-name">${this.escHtml(c.cookie_name)}</span></td>
                    <td>${this.escHtml(c.duration || '—')}</td>
                    <td>${this.escHtml(c.description || '—')}</td>
                </tr>`;
            });

            return `<table class="r2cc-cookie-table">
                <thead><tr><th>${this.escHtml(st.cookieHeader || 'Cookie')}</th><th>${this.escHtml(st.durationHeader || 'Duration')}</th><th>${this.escHtml(st.descHeader || 'Description')}</th></tr></thead>
                <tbody>${rows}</tbody>
            </table>`;
        },

        bindEvents() {
            // Banner buttons
            this.banner.querySelector('[data-action="accept-all"]').addEventListener('click', () => this.acceptAll());
            this.banner.querySelector('[data-action="open-settings"]').addEventListener('click', () => this.openSettings());

            const bannerReject = this.banner.querySelector('[data-action="reject-all"]');
            if (bannerReject) bannerReject.addEventListener('click', () => this.rejectAll());

            // Settings panel buttons
            this.settingsPanel.querySelector('[data-action="accept-all"]').addEventListener('click', () => this.acceptAll());
            this.settingsPanel.querySelector('[data-action="reject-all"]').addEventListener('click', () => this.rejectAll());
            this.settingsPanel.querySelector('[data-action="save-preferences"]').addEventListener('click', () => this.savePreferences());

            // Close settings
            this.settingsPanel.querySelector('.r2cc-settings-close').addEventListener('click', () => this.closeSettings());
            this.settingsPanel.querySelector('.r2cc-settings-overlay').addEventListener('click', () => this.closeSettings());

            // Category accordion toggles
            this.settingsPanel.querySelectorAll('.r2cc-category-header').forEach(header => {
                header.addEventListener('click', (e) => {
                    // Don't toggle if clicking the toggle switch
                    if (e.target.closest('.r2cc-toggle')) return;
                    const item = header.closest('.r2cc-category-item');
                    item.classList.toggle('r2cc-expanded');
                });
            });

            // Re-open button
            if (this.reopenBtn) {
                this.reopenBtn.addEventListener('click', () => this.openSettings());
            }

            // Keyboard: Escape closes settings
            document.addEventListener('keydown', (e) => {
                if (e.key === 'Escape' && this.settingsPanel.style.display !== 'none') {
                    this.closeSettings();
                }
            });
        },

        /* Actions */
        acceptAll() {
            const allCats = Object.keys(C.categories);
            State.setConsent(allCats, 'accept_all');
            ScriptBlocker.unblockAll(allCats);
            GCM.updateConsent(allCats);
            allCats.forEach(cat => this.restoreIframesForCategory(cat));
            this.syncToggles(allCats);
            this.hideBanner();
            this.closeSettings();
            this.showReopen();
        },

        rejectAll() {
            State.setConsent(['necessary'], 'reject_all');
            GCM.updateConsent(['necessary']);
            this.removeCookiesForCategories(['functional', 'analytics', 'marketing']);
            this.syncToggles(['necessary']);
            this.hideBanner();
            this.closeSettings();
            this.showReopen();
        },

        savePreferences() {
            const categories = ['necessary']; // always included
            this.settingsPanel.querySelectorAll('.r2cc-toggle input:checked').forEach(input => {
                const cat = input.getAttribute('data-cat');
                if (cat && !categories.includes(cat)) {
                    categories.push(cat);
                }
            });

            State.setConsent(categories, 'custom');
            ScriptBlocker.unblockAll(categories);
            GCM.updateConsent(categories);
            categories.forEach(cat => this.restoreIframesForCategory(cat));

            // Remove cookies for unchecked categories
            const rejected = Object.keys(C.categories).filter(c => !categories.includes(c));
            this.removeCookiesForCategories(rejected);

            this.hideBanner();
            this.closeSettings();
            this.showReopen();
        },

        /** Sync toggle switches in settings panel to match accepted categories */
        syncToggles(acceptedCategories) {
            if (!this.settingsPanel) return;
            this.settingsPanel.querySelectorAll('.r2cc-toggle input').forEach(input => {
                const cat = input.getAttribute('data-cat');
                if (cat && !input.disabled) {
                    input.checked = acceptedCategories.includes(cat);
                }
            });
        },

        removeCookiesForCategories(categories) {
            categories.forEach(cat => {
                const cookies = C.cookiesByCat[cat] || [];
                cookies.forEach(c => {
                    const name = c.cookie_name.replace(/\*$/, '');
                    // Try to remove exact match
                    Cookies.remove(name);
                    // For wildcard cookies, try removing matching ones
                    if (c.is_regex || c.cookie_name.includes('*')) {
                        document.cookie.split(';').forEach(pair => {
                            const cookieName = pair.split('=')[0].trim();
                            if (cookieName.startsWith(name)) {
                                Cookies.remove(cookieName);
                            }
                        });
                    }
                });
            });
        },

        /* Display helpers */
        showBanner() {
            this.banner.style.display = '';
            if (C.position === 'center-overlay') {
                document.body.classList.add('r2cc-no-scroll');
            }
        },

        hideBanner() {
            this.banner.style.display = 'none';
            document.body.classList.remove('r2cc-no-scroll');
        },

        openSettings() {
            this.hideBanner();
            this.settingsPanel.style.display = '';
            document.body.classList.add('r2cc-no-scroll');

            // Focus trap
            const focusable = this.settingsPanel.querySelectorAll('button, input, a, [tabindex]');
            if (focusable.length) focusable[0].focus();
        },

        closeSettings() {
            this.settingsPanel.style.display = 'none';
            document.body.classList.remove('r2cc-no-scroll');

            // If no consent yet, re-show banner
            if (!State.hasConsent()) {
                this.showBanner();
            }
        },

        showReopen() {
            if (this.reopenBtn) {
                this.reopenBtn.style.display = '';
            }
        },

        escHtml(str) {
            if (!str) return '';
            const div = document.createElement('div');
            div.textContent = str;
            return div.innerHTML;
        },

        /* ============================================================
           IFRAME PLACEHOLDER HANDLING (auto-blocker)
           ============================================================ */
        initIframePlaceholders() {
            // Restore iframes for already-consented categories
            this.restoreIframesForConsented();

            // Handle "Accept & Load" buttons on individual placeholders
            document.addEventListener('click', (e) => {
                const btn = e.target.closest('.r2cc-iframe-accept');
                if (!btn) return;

                const cat = btn.getAttribute('data-r2cc-accept-category');
                if (!cat) return;

                // Accept this category and save consent
                const consent = State.getConsent();
                const currentCats = consent && consent.categories ? [...consent.categories] : ['necessary'];
                if (!currentCats.includes(cat)) {
                    currentCats.push(cat);
                }

                State.setConsent(currentCats, 'custom');
                ScriptBlocker.unblockAll(currentCats);
                GCM.updateConsent(currentCats);

                // Restore all iframes for this category
                this.restoreIframesForCategory(cat);

                // Show reopen button since user made a choice
                this.showReopen();
                this.hideBanner();
            });
        },

        restoreIframesForConsented() {
            const consent = State.getConsent();
            if (!consent || !consent.categories || consent.version !== CONSENT_VERSION) return;

            consent.categories.forEach(cat => {
                this.restoreIframesForCategory(cat);
            });
        },

        restoreIframesForCategory(cat) {
            document.querySelectorAll(`.r2cc-iframe-placeholder[data-r2cc-category="${cat}"]`).forEach(placeholder => {
                const src = placeholder.getAttribute('data-r2cc-src');
                const attrs = placeholder.getAttribute('data-r2cc-attrs') || '';

                if (!src) return;

                const iframe = document.createElement('iframe');
                iframe.src = src;

                // Restore original attributes
                if (attrs) {
                    const temp = document.createElement('div');
                    temp.innerHTML = `<div ${attrs}></div>`;
                    const el = temp.firstChild;
                    for (const attr of el.attributes) {
                        if (attr.name !== 'src') {
                            iframe.setAttribute(attr.name, attr.value);
                        }
                    }
                }

                // Copy dimensions from placeholder
                if (placeholder.style.width) iframe.style.width = placeholder.style.width;
                if (placeholder.style.height) iframe.style.height = placeholder.style.height;

                placeholder.replaceWith(iframe);
            });
        }
    };

    /* ============================================================
       PUBLIC API – window.R2CookieConsent
       ============================================================ */
    window.R2CookieConsent = {
        /** Check if a category is currently consented */
        hasConsent(category) {
            return State.isCategoryAllowed(category);
        },

        /** Get current consent data */
        getConsent() {
            return State.getConsent();
        },

        /** Programmatically accept all */
        acceptAll() {
            UI.acceptAll();
        },

        /** Programmatically reject all */
        rejectAll() {
            UI.rejectAll();
        },

        /** Open settings panel */
        showSettings() {
            UI.openSettings();
        },

        /** Reset consent (clear cookie, show banner again) */
        reset() {
            Cookies.remove(COOKIE_NAME);
            Cookies.remove(UID_COOKIE);
            UI.showBanner();
            if (UI.reopenBtn) UI.reopenBtn.style.display = 'none';
        },

        /** Listen for consent changes */
        on(event, callback) {
            window.addEventListener('r2cc:consent', (e) => callback(e.detail));
        }
    };

    /* ============================================================
       INIT
       ============================================================ */
    function init() {
        ScriptBlocker.init();
        GCM.init();
        UI.init();

        // For returning visitors: check consent version and update GCM
        if (State.hasConsent()) {
            const consent = State.getConsent();
            if (consent && consent.categories) {
                GCM.updateConsent(consent.categories);
                ScriptBlocker.unblockAll(consent.categories);
            }
        }

        // Cache-compatible: if GCM is enabled but the server-side default
        // was served from cache (stale), do a JS-side correction based on cookie
        if (GCM.enabled && typeof gtag === 'function') {
            const consent = State.getConsent();
            if (consent && consent.categories && consent.version === CONSENT_VERSION) {
                // Immediately update with actual consent (overrides cached default)
                GCM.updateConsent(consent.categories);
            }
        }
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
