<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Automatic Script & Iframe Blocker
 *
 * Uses output buffering to intercept the HTML response and:
 * 1. Rewrites known third-party <script> tags to type="text/plain"
 * 2. Replaces known third-party <iframe> elements with placeholders
 * 3. Adds data-r2cc-category attributes for JS-side unblocking
 *
 * This works with ANY cache plugin because it modifies the HTML
 * before it's cached. The JS engine then handles consent checking.
 */
class R2CC_Auto_Blocker {

    /**
     * Pattern database: URL pattern → category
     */
    const SCRIPT_PATTERNS = array(
        // Analytics
        'google-analytics.com'       => 'analytics',
        'googletagmanager.com/gtag'  => 'analytics',
        'analytics.google.com'       => 'analytics',
        'hotjar.com'                 => 'analytics',
        'static.hotjar.com'          => 'analytics',
        'cdn.heapanalytics.com'      => 'analytics',
        'cdn.segment.com'            => 'analytics',
        'cdn.mxpnl.com'             => 'analytics',
        'plausible.io'               => 'analytics',
        'cdn.amplitude.com'          => 'analytics',
        'matomo'                     => 'analytics',
        'clarity.ms'                 => 'analytics',
        'mouseflow.com'              => 'analytics',
        'luckyorange.com'            => 'analytics',
        'fullstory.com'              => 'analytics',
        'crazyegg.com'               => 'analytics',

        // Marketing / Advertising
        'googletagmanager.com/gtm'   => 'marketing',
        'connect.facebook.net'       => 'marketing',
        'fbevents.js'                => 'marketing',
        'snap.licdn.com'             => 'marketing',
        'ads.linkedin.com'           => 'marketing',
        'googleads.g.doubleclick.net' => 'marketing',
        'googlesyndication.com'      => 'marketing',
        'adservice.google'           => 'marketing',
        'amazon-adsystem.com'        => 'marketing',
        'ads-twitter.com'            => 'marketing',
        'static.ads-twitter.com'     => 'marketing',
        'analytics.tiktok.com'       => 'marketing',
        'sc-static.net/scevent'      => 'marketing',
        'pinterest.com/ct'           => 'marketing',
        'ct.pinterest.com'           => 'marketing',
        'tr.snapchat.com'            => 'marketing',
        'bat.bing.com'               => 'marketing',
        'hubspot.com/analytics'      => 'marketing',
        'hs-analytics.net'           => 'marketing',
        'hs-scripts.com'             => 'marketing',
        'js.hs-scripts.com'          => 'marketing',

        // Functional (third-party)
        'widget.intercom.io'         => 'functional',
        'js.intercomcdn.com'         => 'functional',
        'cdn.livechatinc.com'        => 'functional',
        'embed.tawk.to'              => 'functional',
        'crisp.chat'                 => 'functional',
        'js.driftt.com'              => 'functional',
        'zendesk.com/embeddable'     => 'functional',
    );

    /**
     * Iframe patterns: URL pattern → category
     */
    const IFRAME_PATTERNS = array(
        'youtube.com'         => 'marketing',
        'youtube-nocookie.com' => 'marketing',
        'youtu.be'            => 'marketing',
        'player.vimeo.com'    => 'marketing',
        'vimeo.com/video'     => 'marketing',
        'maps.google'         => 'functional',
        'google.com/maps'     => 'functional',
        'facebook.com/plugins' => 'marketing',
        'platform.twitter.com' => 'marketing',
        'instagram.com/embed' => 'marketing',
        'open.spotify.com'    => 'functional',
        'calendly.com'        => 'functional',
        'typeform.com'        => 'functional',
        'hubspot.com'         => 'marketing',
    );

    /**
     * Inline script patterns (for inline gtag/fbq/etc.)
     */
    const INLINE_PATTERNS = array(
        'gtag('          => 'analytics',
        'ga('            => 'analytics',
        '_gaq.push'      => 'analytics',
        'fbq('           => 'marketing',
        '_fbq'           => 'marketing',
        'twq('           => 'marketing',
        'pintrk('        => 'marketing',
        'ttq.load'       => 'marketing',
        'lintrk'         => 'marketing',
        'hj('            => 'analytics',
        '_hjSettings'    => 'analytics',
        'Intercom('      => 'functional',
        'SnapTr.init'    => 'marketing',
    );

    public function __construct() {
        if ( ! Rise2_Cookie_Consent::get_setting( 'auto_block_scripts', 0 ) ) {
            return;
        }

        // Don't block for logged-in admins previewing, bots, or AJAX/REST
        if ( is_admin() || wp_doing_ajax() || wp_doing_cron() ) {
            return;
        }

        if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
            return;
        }

        // Start output buffering on template_redirect (works with all cache plugins)
        add_action( 'template_redirect', array( $this, 'start_buffer' ), 1 );
    }

    /**
     * Start output buffering
     */
    public function start_buffer() {
        ob_start( array( $this, 'process_buffer' ) );
    }

    /**
     * Process the full HTML buffer
     */
    public function process_buffer( $html ) {
        if ( empty( $html ) || ! is_string( $html ) ) {
            return $html;
        }

        // Don't process non-HTML responses
        if ( strpos( $html, '<html' ) === false && strpos( $html, '<!DOCTYPE' ) === false ) {
            return $html;
        }

        // Don't modify admin bar scripts
        $html = $this->block_scripts( $html );
        $html = $this->block_iframes( $html );

        return $html;
    }

    /**
     * Find and block matching <script> tags
     */
    private function block_scripts( $html ) {
        // Match all script tags
        return preg_replace_callback(
            '/<script\b([^>]*)>(.*?)<\/script>/is',
            array( $this, 'process_script_tag' ),
            $html
        );
    }

    /**
     * Process individual script tag
     */
    private function process_script_tag( $matches ) {
        $full_tag   = $matches[0];
        $attributes = $matches[1];
        $content    = $matches[2];

        // Skip if already tagged by us or if it's our own script
        if ( strpos( $attributes, 'data-r2cc-category' ) !== false ) {
            return $full_tag;
        }
        if ( strpos( $attributes, 'r2cc' ) !== false ) {
            return $full_tag;
        }
        // Skip JSON-LD, application/json
        if ( preg_match( '/type\s*=\s*["\'](?:application\/(?:ld\+)?json|text\/template)/i', $attributes ) ) {
            return $full_tag;
        }
        // Don't block GCM default consent script
        if ( strpos( $attributes, 'data-r2cc-gcm' ) !== false ) {
            return $full_tag;
        }

        $category = null;

        // Check external script src
        if ( preg_match( '/src\s*=\s*["\']([^"\']+)["\']/i', $attributes, $src_match ) ) {
            $src = $src_match[1];
            $category = $this->match_url_category( $src, self::SCRIPT_PATTERNS );
        }

        // Check inline script content
        if ( ! $category && ! empty( trim( $content ) ) ) {
            $category = $this->match_inline_category( $content );
        }

        if ( ! $category ) {
            return $full_tag; // Not a known trackable script
        }

        // Rewrite: change type to text/plain, add category
        $new_attrs = $attributes;

        // Replace or add type attribute
        if ( preg_match( '/type\s*=\s*["\'][^"\']*["\']/i', $new_attrs ) ) {
            $new_attrs = preg_replace( '/type\s*=\s*["\'][^"\']*["\']/i', 'type="text/plain"', $new_attrs );
        } else {
            $new_attrs .= ' type="text/plain"';
        }

        $new_attrs .= ' data-r2cc-category="' . esc_attr( $category ) . '"';
        $new_attrs .= ' data-r2cc-blocked="true"';

        return '<script' . $new_attrs . '>' . $content . '</script>';
    }

    /**
     * Find and replace matching <iframe> tags with placeholders
     */
    private function block_iframes( $html ) {
        return preg_replace_callback(
            '/<iframe\b([^>]*)(?:\/>|>(.*?)<\/iframe>)/is',
            array( $this, 'process_iframe_tag' ),
            $html
        );
    }

    /**
     * Process individual iframe tag
     */
    private function process_iframe_tag( $matches ) {
        $full_tag   = $matches[0];
        $attributes = $matches[1];

        // Skip if already handled
        if ( strpos( $attributes, 'data-r2cc-category' ) !== false ) {
            return $full_tag;
        }

        // Get src
        if ( ! preg_match( '/src\s*=\s*["\']([^"\']+)["\']/i', $attributes, $src_match ) ) {
            return $full_tag;
        }

        $src = $src_match[1];
        $category = $this->match_url_category( $src, self::IFRAME_PATTERNS );

        if ( ! $category ) {
            return $full_tag;
        }

        // Extract dimensions
        $width  = '100%';
        $height = '400';
        if ( preg_match( '/width\s*=\s*["\']?(\d+)/i', $attributes, $w ) ) {
            $width = $w[1] . 'px';
        }
        if ( preg_match( '/height\s*=\s*["\']?(\d+)/i', $attributes, $h ) ) {
            $height = $h[1] . 'px';
        }

        // Determine service name for the placeholder
        $service = $this->detect_service_name( $src );

        // Build placeholder HTML
        $placeholder = '<div class="r2cc-iframe-placeholder" '
            . 'data-r2cc-category="' . esc_attr( $category ) . '" '
            . 'data-r2cc-src="' . esc_attr( $src ) . '" '
            . 'data-r2cc-attrs="' . esc_attr( $this->clean_attrs_for_restore( $attributes ) ) . '" '
            . 'style="width:' . esc_attr( $width ) . ';height:' . esc_attr( $height ) . ';">'
            . '<div class="r2cc-iframe-placeholder-inner">'
            . '<svg width="40" height="40" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">'
            . '<path d="M12 22s8-4 8-10V5l-8-3-8 3v7c0 6 8 10 8 10z"/></svg>'
            . '<p class="r2cc-iframe-placeholder-text">'
            . sprintf(
                /* translators: %s: service name like YouTube, Google Maps */
                esc_html__( 'This content requires %s cookies.', 'rise2-cookie-consent' ),
                '<strong>' . esc_html( $service ) . '</strong>'
            )
            . '</p>'
            . '<button type="button" class="r2cc-btn r2cc-btn-accept r2cc-iframe-accept" '
            . 'data-r2cc-accept-category="' . esc_attr( $category ) . '">'
            . esc_html__( 'Accept & Load', 'rise2-cookie-consent' )
            . '</button>'
            . '</div></div>';

        return $placeholder;
    }

    /**
     * Match a URL against known patterns
     */
    private function match_url_category( $url, $patterns ) {
        $url_lower = strtolower( $url );
        foreach ( $patterns as $pattern => $category ) {
            if ( strpos( $url_lower, strtolower( $pattern ) ) !== false ) {
                return $category;
            }
        }
        return null;
    }

    /**
     * Match inline script content against known patterns
     */
    private function match_inline_category( $content ) {
        foreach ( self::INLINE_PATTERNS as $pattern => $category ) {
            if ( strpos( $content, $pattern ) !== false ) {
                return $category;
            }
        }
        return null;
    }

    /**
     * Detect the service name from URL for placeholder text
     */
    private function detect_service_name( $url ) {
        $map = array(
            'youtube'   => 'YouTube',
            'vimeo'     => 'Vimeo',
            'google.com/maps' => 'Google Maps',
            'maps.google' => 'Google Maps',
            'facebook'  => 'Facebook',
            'twitter'   => 'Twitter',
            'instagram' => 'Instagram',
            'spotify'   => 'Spotify',
            'calendly'  => 'Calendly',
            'typeform'  => 'Typeform',
            'hubspot'   => 'HubSpot',
        );

        $url_lower = strtolower( $url );
        foreach ( $map as $pattern => $name ) {
            if ( strpos( $url_lower, $pattern ) !== false ) {
                return $name;
            }
        }

        // Extract domain as fallback
        $host = parse_url( $url, PHP_URL_HOST );
        return $host ?: __( 'third-party', 'rise2-cookie-consent' );
    }

    /**
     * Clean iframe attributes for data attribute storage (for restoring later)
     */
    private function clean_attrs_for_restore( $attrs ) {
        // Remove src (we store it separately) and clean up
        $attrs = preg_replace( '/src\s*=\s*["\'][^"\']*["\']/i', '', $attrs );
        return trim( preg_replace( '/\s+/', ' ', $attrs ) );
    }

    /**
     * Get all patterns for admin display
     */
    public static function get_all_patterns() {
        return array(
            'scripts'  => self::SCRIPT_PATTERNS,
            'iframes'  => self::IFRAME_PATTERNS,
            'inline'   => self::INLINE_PATTERNS,
        );
    }
}
