<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class R2CC_Consent_Logger {

    /**
     * Log a consent action
     */
    public static function log_consent( $categories, $consent_type = 'custom' ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        // Generate or retrieve consent UID
        $consent_uid = isset( $_COOKIE['r2cc_consent_uid'] ) 
            ? sanitize_text_field( $_COOKIE['r2cc_consent_uid'] ) 
            : self::generate_uid();

        // Anonymize IP (store only first 3 octets for IPv4, first 3 groups for IPv6)
        $ip = self::anonymize_ip( self::get_client_ip() );

        $wpdb->insert(
            $table,
            array(
                'consent_uid'  => $consent_uid,
                'consent_type' => $consent_type,
                'categories'   => wp_json_encode( $categories ),
                'ip_address'   => $ip,
                'user_agent'   => substr( sanitize_text_field( $_SERVER['HTTP_USER_AGENT'] ?? '' ), 0, 500 ),
                'created_at'   => current_time( 'mysql' ),
            ),
            array( '%s', '%s', '%s', '%s', '%s', '%s' )
        );

        return $consent_uid;
    }

    /**
     * Get consent logs with pagination
     */
    public static function get_logs( $page = 1, $per_page = 50 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';
        $offset = ( $page - 1 ) * $per_page;

        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

        $logs = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM {$table} ORDER BY created_at DESC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            )
        );

        return array(
            'logs'       => $logs,
            'total'      => intval( $total ),
            'pages'      => ceil( $total / $per_page ),
            'current'    => $page,
        );
    }

    /**
     * Get consent statistics
     */
    public static function get_stats() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );

        $by_type = $wpdb->get_results(
            "SELECT consent_type, COUNT(*) as count FROM {$table} GROUP BY consent_type"
        );

        $last_30_days = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM {$table} WHERE created_at >= %s",
                date( 'Y-m-d H:i:s', strtotime( '-30 days' ) )
            )
        );

        return array(
            'total'          => intval( $total ),
            'by_type'        => $by_type,
            'last_30_days'   => intval( $last_30_days ),
        );
    }

    /**
     * Get daily consent counts for the last N days (for chart)
     */
    public static function get_daily_counts( $days = 30 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT DATE(created_at) as date, consent_type, COUNT(*) as count
                 FROM {$table}
                 WHERE created_at >= %s
                 GROUP BY DATE(created_at), consent_type
                 ORDER BY date ASC",
                date( 'Y-m-d', strtotime( "-{$days} days" ) )
            )
        );

        // Build a complete date range with zeros
        $daily = array();
        $start = new DateTime( date( 'Y-m-d', strtotime( "-{$days} days" ) ) );
        $end   = new DateTime( date( 'Y-m-d' ) );
        $end->modify( '+1 day' );

        $interval = new DateInterval( 'P1D' );
        $range    = new DatePeriod( $start, $interval, $end );

        foreach ( $range as $dt ) {
            $d = $dt->format( 'Y-m-d' );
            $daily[ $d ] = array(
                'date'       => $d,
                'accept_all' => 0,
                'reject_all' => 0,
                'custom'     => 0,
                'total'      => 0,
            );
        }

        foreach ( $results as $row ) {
            if ( isset( $daily[ $row->date ] ) ) {
                $type = $row->consent_type;
                if ( isset( $daily[ $row->date ][ $type ] ) ) {
                    $daily[ $row->date ][ $type ] = intval( $row->count );
                }
                $daily[ $row->date ]['total'] += intval( $row->count );
            }
        }

        return array_values( $daily );
    }

    /**
     * Get category acceptance rates (how many users accepted each category)
     */
    public static function get_category_rates() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
        if ( ! $total ) {
            return array();
        }

        $all_logs = $wpdb->get_col( "SELECT categories FROM {$table}" );

        $counts = array(
            'necessary'  => 0,
            'functional' => 0,
            'analytics'  => 0,
            'marketing'  => 0,
        );

        foreach ( $all_logs as $cats_json ) {
            $cats = json_decode( $cats_json, true );
            if ( ! is_array( $cats ) ) continue;
            foreach ( $cats as $cat ) {
                if ( isset( $counts[ $cat ] ) ) {
                    $counts[ $cat ]++;
                }
            }
        }

        $rates = array();
        foreach ( $counts as $cat => $count ) {
            $rates[ $cat ] = array(
                'count' => $count,
                'rate'  => round( ( $count / $total ) * 100, 1 ),
            );
        }

        return $rates;
    }

    /**
     * Get consent type breakdown (accept/reject/custom percentages)
     */
    public static function get_type_breakdown() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        $total = $wpdb->get_var( "SELECT COUNT(*) FROM {$table}" );
        if ( ! $total ) {
            return array();
        }

        $types = $wpdb->get_results(
            "SELECT consent_type, COUNT(*) as count FROM {$table} GROUP BY consent_type"
        );

        $breakdown = array();
        foreach ( $types as $t ) {
            $breakdown[ $t->consent_type ] = array(
                'count' => intval( $t->count ),
                'rate'  => round( ( $t->count / $total ) * 100, 1 ),
            );
        }

        return $breakdown;
    }

    /**
     * Export consent logs as CSV
     */
    public static function export_csv() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        $logs = $wpdb->get_results( "SELECT * FROM {$table} ORDER BY created_at DESC", ARRAY_A );

        $filename = 'consent-logs-' . date( 'Y-m-d' ) . '.csv';

        header( 'Content-Type: text/csv; charset=utf-8' );
        header( 'Content-Disposition: attachment; filename=' . $filename );

        $output = fopen( 'php://output', 'w' );
        fputcsv( $output, array( 'ID', 'Consent UID', 'Type', 'Categories', 'IP (anonymized)', 'Date' ) );

        foreach ( $logs as $log ) {
            fputcsv( $output, array(
                $log['id'],
                $log['consent_uid'],
                $log['consent_type'],
                $log['categories'],
                $log['ip_address'],
                $log['created_at'],
            ) );
        }

        fclose( $output );
        exit;
    }

    /**
     * Purge old consent logs
     */
    public static function purge_old_logs( $days = 365 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_consent_logs';

        return $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$table} WHERE created_at < %s",
                date( 'Y-m-d H:i:s', strtotime( "-{$days} days" ) )
            )
        );
    }

    /**
     * Generate unique consent ID
     */
    private static function generate_uid() {
        return wp_generate_uuid4();
    }

    /**
     * Anonymize IP address for GDPR compliance
     */
    private static function anonymize_ip( $ip ) {
        if ( strpos( $ip, ':' ) !== false ) {
            // IPv6: zero out last 80 bits
            return preg_replace( '/:[0-9a-f]*:[0-9a-f]*:[0-9a-f]*:[0-9a-f]*:[0-9a-f]*$/i', ':0:0:0:0:0', $ip );
        } else {
            // IPv4: zero out last octet
            return preg_replace( '/\.\d+$/', '.0', $ip );
        }
    }

    /**
     * Get client IP address
     */
    private static function get_client_ip() {
        $ip_keys = array(
            'HTTP_CF_CONNECTING_IP',  // Cloudflare
            'HTTP_X_FORWARDED_FOR',
            'HTTP_X_REAL_IP',
            'REMOTE_ADDR',
        );

        foreach ( $ip_keys as $key ) {
            if ( ! empty( $_SERVER[ $key ] ) ) {
                $ip = explode( ',', $_SERVER[ $key ] );
                $ip = trim( $ip[0] );
                if ( filter_var( $ip, FILTER_VALIDATE_IP ) ) {
                    return $ip;
                }
            }
        }

        return '0.0.0.0';
    }
}
