<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class R2CC_Cookie_Database {

    /**
     * Create database tables on activation
     */
    public static function create_tables() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $cookies_table = $wpdb->prefix . 'r2cc_cookies';
        $consent_table = $wpdb->prefix . 'r2cc_consent_logs';

        $sql = "CREATE TABLE {$cookies_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            cookie_name varchar(255) NOT NULL,
            category varchar(50) NOT NULL DEFAULT 'necessary',
            domain varchar(255) DEFAULT '',
            duration varchar(100) DEFAULT '',
            description text DEFAULT '',
            is_regex tinyint(1) DEFAULT 0,
            script_url text DEFAULT '',
            is_auto_detected tinyint(1) DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_category (category),
            KEY idx_cookie_name (cookie_name(191))
        ) {$charset_collate};

        CREATE TABLE {$consent_table} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            consent_uid varchar(64) NOT NULL,
            consent_type varchar(20) NOT NULL DEFAULT 'custom',
            categories text NOT NULL,
            ip_address varchar(45) DEFAULT '',
            user_agent text DEFAULT '',
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY idx_consent_uid (consent_uid),
            KEY idx_created_at (created_at)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );

        update_option( 'r2cc_db_version', R2CC_VERSION );
    }

    /**
     * Get all cookies, optionally filtered by category
     */
    public static function get_cookies( $category = null ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';

        if ( $category ) {
            return $wpdb->get_results(
                $wpdb->prepare( "SELECT * FROM {$table} WHERE category = %s ORDER BY cookie_name ASC", $category )
            );
        }

        return $wpdb->get_results( "SELECT * FROM {$table} ORDER BY category ASC, cookie_name ASC" );
    }

    /**
     * Get a single cookie by ID
     */
    public static function get_cookie( $id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';
        return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE id = %d", $id ) );
    }

    /**
     * Save or update a cookie
     */
    public static function save_cookie( $data, $id = 0 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';

        $fields = array(
            'cookie_name' => sanitize_text_field( $data['cookie_name'] ?? '' ),
            'category'    => sanitize_text_field( $data['category'] ?? 'necessary' ),
            'domain'      => sanitize_text_field( $data['domain'] ?? '' ),
            'duration'    => sanitize_text_field( $data['duration'] ?? '' ),
            'description' => sanitize_textarea_field( $data['description'] ?? '' ),
            'is_regex'    => intval( $data['is_regex'] ?? 0 ),
            'script_url'  => esc_url_raw( $data['script_url'] ?? '' ),
        );

        if ( $id > 0 ) {
            $wpdb->update( $table, $fields, array( 'id' => $id ), null, array( '%d' ) );
            return $id;
        } else {
            $wpdb->insert( $table, $fields );
            return $wpdb->insert_id;
        }
    }

    /**
     * Delete a cookie entry
     */
    public static function delete_cookie( $id ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';
        return $wpdb->delete( $table, array( 'id' => intval( $id ) ), array( '%d' ) );
    }

    /**
     * Check if a cookie name already exists
     */
    public static function cookie_exists( $name ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';
        return $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$table} WHERE cookie_name = %s", $name ) ) > 0;
    }

    /**
     * Get cookie count by category
     */
    public static function get_cookie_counts() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';
        $results = $wpdb->get_results( "SELECT category, COUNT(*) as count FROM {$table} GROUP BY category" );

        $counts = array();
        foreach ( $results as $row ) {
            $counts[ $row->category ] = intval( $row->count );
        }
        return $counts;
    }

    /**
     * Bulk import cookies (used by scanner)
     */
    public static function bulk_import( $cookies ) {
        $imported = 0;
        foreach ( $cookies as $cookie ) {
            if ( ! self::cookie_exists( $cookie['cookie_name'] ) ) {
                $cookie['is_auto_detected'] = 1;
                self::save_cookie( $cookie );
                $imported++;
            }
        }
        return $imported;
    }

    /**
     * Get all cookies (no filter)
     */
    public static function get_all_cookies() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';
        return $wpdb->get_results( "SELECT * FROM {$table} ORDER BY category ASC, cookie_name ASC" );
    }

    /**
     * Get a single cookie by name
     */
    public static function get_cookie_by_name( $name ) {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';
        return $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$table} WHERE cookie_name = %s", $name ) );
    }
}
