<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class R2CC_Cookie_Scanner {

    /**
     * Known cookies database - comprehensive list of common cookies
     */
    private $known_cookies = array();

    public function __construct() {
        $this->known_cookies = $this->get_known_cookies_db();
    }

    /**
     * Perform a scan of the site (server-side detection)
     */
    public function scan() {
        $detected = array();

        // 1. Detect our own consent cookies
        $detected = array_merge( $detected, $this->scan_wordpress_core() );

        // 2. Detect active plugin cookies (curated database)
        $detected = array_merge( $detected, $this->scan_active_plugins() );

        // 3. Detect scripts in page source (regex patterns)
        $detected = array_merge( $detected, $this->scan_scripts_in_pages() );

        // 4. Import detected cookies into DB
        $imported = R2CC_Cookie_Database::bulk_import( $detected );

        return array(
            'detected'  => count( $detected ),
            'imported'  => $imported,
            'cookies'   => $detected,
            'timestamp' => current_time( 'mysql' ),
        );
    }

    /**
     * Process real cookies reported by the admin's browser.
     * Called via AJAX after the frontend loads in a hidden iframe.
     *
     * @param array $browser_cookies Array of cookie name strings from document.cookie
     * @return array Import results
     */
    public function process_browser_cookies( $browser_cookies ) {
        if ( empty( $browser_cookies ) || ! is_array( $browser_cookies ) ) {
            return array( 'detected' => 0, 'imported' => 0, 'new_cookies' => array() );
        }

        $domain = parse_url( home_url(), PHP_URL_HOST );
        $known = $this->get_known_cookies_db();
        $to_import = array();
        $new_unknown = array();

        foreach ( $browser_cookies as $name ) {
            $name = sanitize_text_field( trim( $name ) );
            if ( empty( $name ) ) continue;

            // Skip our own cookies
            if ( strpos( $name, 'r2cc_' ) === 0 ) continue;

            // Skip WordPress admin cookies
            if ( $this->is_admin_cookie( $name ) ) continue;

            // Try to match against known cookie database
            $match = $this->match_known_cookie( $name, $known );

            if ( $match ) {
                $to_import[] = $match;
            } else {
                // Unknown cookie — flag for review
                $cookie_data = array(
                    'cookie_name' => $name,
                    'category'    => 'functional', // safe default
                    'domain'      => $domain,
                    'duration'    => 'Unknown',
                    'description' => 'Automatically detected by browser scan. Please review and categorize.',
                );
                $to_import[] = $cookie_data;
                $new_unknown[] = $name;
            }
        }

        $imported = R2CC_Cookie_Database::bulk_import( $to_import );

        return array(
            'detected'    => count( $to_import ),
            'imported'    => $imported,
            'new_cookies' => $new_unknown,
        );
    }

    /**
     * Check if a cookie name is an admin-only cookie we should skip.
     */
    private function is_admin_cookie( $name ) {
        $admin_prefixes = array(
            'wordpress_logged_in_',
            'wordpress_sec_',
            'wordpress_test_cookie',
            'wp-settings-',
            'wp-settings-time-',
            'comment_author_',
            'comment_author_email_',
            'yoast-seo-',
        );

        foreach ( $admin_prefixes as $prefix ) {
            if ( strpos( $name, $prefix ) === 0 || $name === $prefix ) {
                return true;
            }
        }

        return false;
    }

    /**
     * Try to match a cookie name against the known cookies database.
     * Supports exact match and wildcard patterns.
     */
    private function match_known_cookie( $name, $known ) {
        foreach ( $known as $entry ) {
            $pattern = $entry['cookie_name'];

            // Exact match
            if ( $pattern === $name ) {
                return $entry;
            }

            // Wildcard match (e.g., _ga_* matches _ga_ABC123)
            if ( substr( $pattern, -1 ) === '*' ) {
                $prefix = substr( $pattern, 0, -1 );
                if ( strpos( $name, $prefix ) === 0 ) {
                    $entry['cookie_name'] = $name; // Use actual name
                    return $entry;
                }
            }
        }

        return null;
    }

    /**
     * Scan for WordPress core cookies
     */
    private function scan_wordpress_core() {
        // Only include cookies that regular (non-admin) visitors encounter.
        // Excluded: wordpress_logged_in_*, wordpress_sec_*, wordpress_test_cookie,
        // wp-settings-*, wp-settings-time-*, comment_author_*, comment_author_email_*
        // These are either admin-only or rarely seen by typical visitors.
        $cookies = array();

        // Our own consent cookie — always present
        $cookies[] = array(
            'cookie_name' => 'r2cc_consent',
            'category'    => 'necessary',
            'domain'      => parse_url( home_url(), PHP_URL_HOST ),
            'duration'    => get_option( 'r2cc_consent_expiry_days', 365 ) . ' days',
            'description' => 'Stores the visitor\'s cookie consent preferences (which categories were accepted or rejected).',
        );
        $cookies[] = array(
            'cookie_name' => 'r2cc_consent_uid',
            'category'    => 'necessary',
            'domain'      => parse_url( home_url(), PHP_URL_HOST ),
            'duration'    => get_option( 'r2cc_consent_expiry_days', 365 ) . ' days',
            'description' => 'Anonymous identifier used to record the consent action for compliance audit purposes.',
        );

        return $cookies;
    }

    /**
     * Detect cookies from active plugins
     */
    private function scan_active_plugins() {
        $detected = array();
        $active_plugins = get_option( 'active_plugins', array() );
        $plugin_text = implode( '|', $active_plugins );
        $domain = parse_url( home_url(), PHP_URL_HOST );

        // WooCommerce
        if ( class_exists( 'WooCommerce' ) || stripos( $plugin_text, 'woocommerce' ) !== false ) {
            $detected = array_merge( $detected, $this->get_woocommerce_cookies() );
        }

        // Google Analytics / Google Tag Manager (via various plugins)
        if ( $this->has_active_plugin( $active_plugins, array( 'google-analytics', 'google-site-kit', 'gtm', 'monsterinsights', 'analytify', 'ga-google-analytics' ) ) ) {
            $detected = array_merge( $detected, $this->get_google_analytics_cookies() );
        }

        // Facebook Pixel
        if ( $this->has_active_plugin( $active_plugins, array( 'facebook', 'fb-pixel', 'pixelyoursite' ) ) ) {
            $detected = array_merge( $detected, $this->get_facebook_cookies() );
        }

        // Contact Form 7
        if ( $this->has_active_plugin( $active_plugins, array( 'contact-form-7' ) ) ) {
            $detected[] = array(
                'cookie_name' => 'wp_cf7_*',
                'category'    => 'functional',
                'domain'      => $domain,
                'duration'    => 'Session',
                'description' => 'Used by Contact Form 7 to prevent spam and track form submissions.',
                'is_regex'    => 1,
            );
        }

        // WPForms
        if ( $this->has_active_plugin( $active_plugins, array( 'wpforms' ) ) ) {
            $detected[] = array(
                'cookie_name' => 'wpforms_*',
                'category'    => 'functional',
                'domain'      => $domain,
                'duration'    => 'Session',
                'description' => 'Used by WPForms to store form state and anti-spam tokens.',
                'is_regex'    => 1,
            );
        }

        // Gravity Forms
        if ( $this->has_active_plugin( $active_plugins, array( 'gravityforms' ) ) ) {
            $detected[] = array(
                'cookie_name' => 'gf_*',
                'category'    => 'functional',
                'domain'      => $domain,
                'duration'    => 'Session',
                'description' => 'Used by Gravity Forms to manage form submissions and prevent duplicate entries.',
                'is_regex'    => 1,
            );
        }

        // Elementor
        if ( $this->has_active_plugin( $active_plugins, array( 'elementor' ) ) ) {
            $detected[] = array(
                'cookie_name' => 'elementor',
                'category'    => 'functional',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => 'Persistent',
                'description' => 'Used by the Elementor page builder to remember visitor actions and preferences.',
            );
        }

        // WPML
        if ( $this->has_active_plugin( $active_plugins, array( 'sitepress-multilingual', 'wpml' ) ) ) {
            $detected[] = array(
                'cookie_name' => 'wp-wpml_current_language',
                'category'    => 'functional',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => 'Session',
                'description' => 'Stores the current language selected by the user on the multilingual website.',
            );
            $detected[] = array(
                'cookie_name' => '_icl_current_language',
                'category'    => 'functional',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '1 day',
                'description' => 'Stores the language preference for the WPML multilingual plugin.',
            );
        }

        // Yoast SEO — admin-only cookies, excluded
        // yoast-seo-* cookies are only set in wp-admin

        return $detected;
    }

    /**
     * Scan page source for embedded scripts that set cookies
     */
    private function scan_scripts_in_pages() {
        $detected = array();

        // Get the homepage HTML
        $response = wp_remote_get( home_url(), array(
            'timeout'   => 15,
            'sslverify' => false,
        ) );

        if ( is_wp_error( $response ) ) {
            return $detected;
        }

        $body = wp_remote_retrieve_body( $response );

        // Google Analytics
        if ( preg_match( '/gtag|google-analytics|googletagmanager|ga\s*\(/i', $body ) ) {
            $detected = array_merge( $detected, $this->get_google_analytics_cookies() );
        }

        // Google Tag Manager
        if ( preg_match( '/googletagmanager\.com\/gtm/i', $body ) ) {
            $detected[] = array(
                'cookie_name' => '_gcl_au',
                'category'    => 'analytics',
                'domain'      => '.google.com',
                'duration'    => '90 days',
                'description' => 'Used by Google Tag Manager to track ad efficiency and conversions.',
            );
        }

        // Facebook Pixel
        if ( preg_match( '/fbq|facebook\.net\/en_US\/fbevents/i', $body ) ) {
            $detected = array_merge( $detected, $this->get_facebook_cookies() );
        }

        // Hotjar
        if ( preg_match( '/hotjar|hj\./i', $body ) ) {
            $detected = array_merge( $detected, array(
                array(
                    'cookie_name' => '_hj*',
                    'category'    => 'analytics',
                    'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                    'duration'    => '365 days',
                    'description' => 'Hotjar cookies used for analytics, heatmaps, and session recordings to understand user behavior.',
                    'is_regex'    => 1,
                ),
            ));
        }

        // YouTube embeds
        if ( preg_match( '/youtube\.com|youtu\.be|youtube-nocookie\.com/i', $body ) ) {
            $detected = array_merge( $detected, array(
                array(
                    'cookie_name' => 'YSC',
                    'category'    => 'marketing',
                    'domain'      => '.youtube.com',
                    'duration'    => 'Session',
                    'description' => 'Set by YouTube to track views of embedded videos.',
                ),
                array(
                    'cookie_name' => 'VISITOR_INFO1_LIVE',
                    'category'    => 'marketing',
                    'domain'      => '.youtube.com',
                    'duration'    => '180 days',
                    'description' => 'Used by YouTube to estimate user bandwidth and adjust video quality.',
                ),
            ));
        }

        // Google Maps
        if ( preg_match( '/maps\.googleapis|maps\.google/i', $body ) ) {
            $detected[] = array(
                'cookie_name' => 'NID',
                'category'    => 'functional',
                'domain'      => '.google.com',
                'duration'    => '6 months',
                'description' => 'Set by Google Maps to store user preferences and information.',
            );
        }

        // Vimeo
        if ( preg_match( '/vimeo\.com/i', $body ) ) {
            $detected[] = array(
                'cookie_name' => 'vuid',
                'category'    => 'analytics',
                'domain'      => '.vimeo.com',
                'duration'    => '2 years',
                'description' => 'Set by Vimeo to collect tracking information for embedded videos.',
            );
        }

        // HubSpot
        if ( preg_match( '/hubspot|hs-script/i', $body ) ) {
            $detected = array_merge( $detected, array(
                array(
                    'cookie_name' => '__hssc',
                    'category'    => 'analytics',
                    'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                    'duration'    => '30 minutes',
                    'description' => 'HubSpot cookie that tracks sessions and pageviews.',
                ),
                array(
                    'cookie_name' => '__hssrc',
                    'category'    => 'analytics',
                    'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                    'duration'    => 'Session',
                    'description' => 'HubSpot cookie that determines if the user has restarted their browser.',
                ),
                array(
                    'cookie_name' => '__hstc',
                    'category'    => 'analytics',
                    'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                    'duration'    => '13 months',
                    'description' => 'HubSpot main tracking cookie used to identify visitors.',
                ),
                array(
                    'cookie_name' => 'hubspotutk',
                    'category'    => 'analytics',
                    'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                    'duration'    => '13 months',
                    'description' => 'Keeps track of a visitor identity. Used for HubSpot CRM integration.',
                ),
            ));
        }

        // Intercom
        if ( preg_match( '/intercom/i', $body ) ) {
            $detected[] = array(
                'cookie_name' => 'intercom-*',
                'category'    => 'functional',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '9 months',
                'description' => 'Used by Intercom live chat to identify users and store session data.',
                'is_regex'    => 1,
            );
        }

        // Stripe
        if ( preg_match( '/stripe\.com|stripe\.js/i', $body ) ) {
            $detected[] = array(
                'cookie_name' => '__stripe_*',
                'category'    => 'necessary',
                'domain'      => '.stripe.com',
                'duration'    => '2 years',
                'description' => 'Set by Stripe payment processor to facilitate secure payment transactions and fraud prevention.',
                'is_regex'    => 1,
            );
        }

        // LinkedIn Insight Tag
        if ( preg_match( '/snap\.licdn|linkedin\.com\/px/i', $body ) ) {
            $detected = array_merge( $detected, array(
                array(
                    'cookie_name' => 'li_sugr',
                    'category'    => 'marketing',
                    'domain'      => '.linkedin.com',
                    'duration'    => '90 days',
                    'description' => 'LinkedIn browser identifier cookie used for tracking and advertising.',
                ),
                array(
                    'cookie_name' => 'bcookie',
                    'category'    => 'marketing',
                    'domain'      => '.linkedin.com',
                    'duration'    => '1 year',
                    'description' => 'LinkedIn browser ID cookie used to identify browser instances for analytics.',
                ),
            ));
        }

        // Deduplicate by cookie_name
        $seen = array();
        $unique = array();
        foreach ( $detected as $cookie ) {
            if ( ! in_array( $cookie['cookie_name'], $seen ) ) {
                $seen[] = $cookie['cookie_name'];
                $unique[] = $cookie;
            }
        }

        return $unique;
    }

    /**
     * WooCommerce cookies
     */
    private function get_woocommerce_cookies() {
        $domain = parse_url( home_url(), PHP_URL_HOST );
        return array(
            array(
                'cookie_name' => 'woocommerce_cart_hash',
                'category'    => 'necessary',
                'domain'      => $domain,
                'duration'    => 'Session',
                'description' => 'Stores a hash of the shopping cart contents to quickly detect changes.',
            ),
            array(
                'cookie_name' => 'woocommerce_items_in_cart',
                'category'    => 'necessary',
                'domain'      => $domain,
                'duration'    => 'Session',
                'description' => 'Tracks whether items are in the shopping cart for cache management.',
            ),
            array(
                'cookie_name' => 'wp_woocommerce_session_*',
                'category'    => 'necessary',
                'domain'      => $domain,
                'duration'    => '2 days',
                'description' => 'Contains a unique identifier for the shopping session to maintain cart and checkout data.',
                'is_regex'    => 1,
            ),
            array(
                'cookie_name' => 'wc_cart_created',
                'category'    => 'necessary',
                'domain'      => $domain,
                'duration'    => 'Session',
                'description' => 'Stores the timestamp when the shopping cart was first created.',
            ),
        );
    }

    /**
     * Google Analytics cookies
     */
    private function get_google_analytics_cookies() {
        return array(
            array(
                'cookie_name' => '_ga',
                'category'    => 'analytics',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '2 years',
                'description' => 'Google Analytics cookie used to distinguish unique users by assigning a randomly generated number as a client identifier.',
            ),
            array(
                'cookie_name' => '_ga_*',
                'category'    => 'analytics',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '2 years',
                'description' => 'Used by Google Analytics 4 to persist session state and track interactions.',
                'is_regex'    => 1,
            ),
            array(
                'cookie_name' => '_gid',
                'category'    => 'analytics',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '24 hours',
                'description' => 'Google Analytics cookie used to distinguish users. Stores and updates a unique value for each page visited.',
            ),
            array(
                'cookie_name' => '_gat',
                'category'    => 'analytics',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '1 minute',
                'description' => 'Used by Google Analytics to throttle the request rate and limit data collection on high traffic sites.',
            ),
        );
    }

    /**
     * Facebook/Meta cookies
     */
    private function get_facebook_cookies() {
        return array(
            array(
                'cookie_name' => '_fbp',
                'category'    => 'marketing',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '90 days',
                'description' => 'Used by Facebook to deliver advertising and track ad performance when users interact with the website.',
            ),
            array(
                'cookie_name' => '_fbc',
                'category'    => 'marketing',
                'domain'      => parse_url( home_url(), PHP_URL_HOST ),
                'duration'    => '90 days',
                'description' => 'Facebook click identifier cookie that stores the last ad click information for conversion tracking.',
            ),
            array(
                'cookie_name' => 'fr',
                'category'    => 'marketing',
                'domain'      => '.facebook.com',
                'duration'    => '90 days',
                'description' => 'Used by Facebook for advertising, delivering, and measuring the relevancy of ads.',
            ),
        );
    }

    /**
     * Check if any matching plugin is active
     */
    private function has_active_plugin( $active_plugins, $search_terms ) {
        $plugin_text = implode( '|', $active_plugins );
        foreach ( $search_terms as $term ) {
            if ( stripos( $plugin_text, $term ) !== false ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Get the known cookies database for matching
     */
    private function get_known_cookies_db() {
        return array(
            '_ga'       => array( 'category' => 'analytics', 'description' => 'Google Analytics tracking cookie' ),
            '_gid'      => array( 'category' => 'analytics', 'description' => 'Google Analytics session cookie' ),
            '_fbp'      => array( 'category' => 'marketing', 'description' => 'Facebook pixel tracking cookie' ),
            '_hj*'      => array( 'category' => 'analytics', 'description' => 'Hotjar analytics cookie' ),
        );
    }
}
