<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Scheduled Cookie Re-Scanner
 *
 * Uses WP-Cron to periodically scan for new cookies and
 * notify admin if unrecognized cookies are detected.
 */
class R2CC_Cron {

    const HOOK_NAME     = 'r2cc_scheduled_scan';
    const TRANSIENT_KEY = 'r2cc_scan_alert';

    public function __construct() {
        add_action( self::HOOK_NAME, array( $this, 'run_scan' ) );
        add_action( 'admin_notices', array( $this, 'show_scan_notice' ) );
        add_action( 'wp_ajax_r2cc_dismiss_scan_alert', array( $this, 'dismiss_alert' ) );

        // Register custom interval
        add_filter( 'cron_schedules', array( $this, 'add_cron_schedule' ) );
    }

    /**
     * Add custom cron schedule
     */
    public function add_cron_schedule( $schedules ) {
        $schedules['r2cc_weekly'] = array(
            'interval' => 604800,
            'display'  => __( 'Weekly (RISE2 Cookie Scan)', 'rise2-cookie-consent' ),
        );
        $schedules['r2cc_daily'] = array(
            'interval' => 86400,
            'display'  => __( 'Daily (RISE2 Cookie Scan)', 'rise2-cookie-consent' ),
        );
        return $schedules;
    }

    /**
     * Schedule the scan event
     */
    public static function schedule( $frequency = 'r2cc_weekly' ) {
        self::unschedule(); // Clear existing first
        if ( $frequency !== 'disabled' ) {
            wp_schedule_event( time() + 3600, $frequency, self::HOOK_NAME );
        }
    }

    /**
     * Unschedule the scan event
     */
    public static function unschedule() {
        $timestamp = wp_next_scheduled( self::HOOK_NAME );
        if ( $timestamp ) {
            wp_unschedule_event( $timestamp, self::HOOK_NAME );
        }
    }

    /**
     * Get next scheduled scan time
     */
    public static function get_next_scan() {
        return wp_next_scheduled( self::HOOK_NAME );
    }

    /**
     * Get last scan results
     */
    public static function get_last_scan() {
        return get_option( 'r2cc_last_scan', null );
    }

    /**
     * Run the scheduled scan
     */
    public function run_scan() {
        $scanner = new R2CC_Cookie_Scanner();
        $results = $scanner->scan();

        // Store scan results
        $scan_data = array(
            'timestamp'  => current_time( 'mysql' ),
            'detected'   => $results['detected'],
            'imported'   => $results['imported'],
            'cookies'    => $results['cookies'],
        );

        update_option( 'r2cc_last_scan', $scan_data );

        // If new cookies were imported, set an admin alert
        if ( $results['imported'] > 0 ) {
            set_transient( self::TRANSIENT_KEY, array(
                'imported' => $results['imported'],
                'time'     => current_time( 'mysql' ),
            ), 30 * DAY_IN_SECONDS );

            // Send email notification if enabled
            if ( Rise2_Cookie_Consent::get_setting( 'scan_email_notify', 0 ) ) {
                $this->send_notification( $results );
            }
        }
    }

    /**
     * Show admin notice when new cookies are detected
     */
    public function show_scan_notice() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $alert = get_transient( self::TRANSIENT_KEY );
        if ( ! $alert ) {
            return;
        }

        $settings_url = admin_url( 'admin.php?page=r2cc-settings&tab=cookies' );
        ?>
        <div class="notice notice-warning is-dismissible r2cc-scan-notice" data-dismiss-action="r2cc_dismiss_scan_alert">
            <p>
                <strong>🍪 <?php esc_html_e( 'RISE2 Cookie Consent:', 'rise2-cookie-consent' ); ?></strong>
                <?php printf(
                    /* translators: 1: number of cookies, 2: settings link */
                    esc_html__( 'Automatic scan detected %1$d new cookie(s) on your site. Please %2$s to review and categorize them.', 'rise2-cookie-consent' ),
                    intval( $alert['imported'] ),
                    '<a href="' . esc_url( $settings_url ) . '">' . esc_html__( 'visit Cookie settings', 'rise2-cookie-consent' ) . '</a>'
                ); ?>
            </p>
        </div>
        <script>
        jQuery(function($){
            $(document).on('click', '.r2cc-scan-notice .notice-dismiss', function(){
                $.post(ajaxurl, { action: 'r2cc_dismiss_scan_alert', nonce: '<?php echo wp_create_nonce('r2cc_dismiss_scan'); ?>' });
            });
        });
        </script>
        <?php
    }

    /**
     * Dismiss the scan alert
     */
    public function dismiss_alert() {
        check_ajax_referer( 'r2cc_dismiss_scan', 'nonce' );
        delete_transient( self::TRANSIENT_KEY );
        wp_send_json_success();
    }

    /**
     * Send email notification about new cookies
     */
    private function send_notification( $results ) {
        $admin_email = get_option( 'admin_email' );
        $site_name   = get_bloginfo( 'name' );
        $settings_url = admin_url( 'admin.php?page=r2cc-settings&tab=cookies' );

        $subject = sprintf(
            /* translators: 1: site name, 2: number of cookies */
            __( '[%1$s] %2$d new cookie(s) detected', 'rise2-cookie-consent' ),
            $site_name,
            $results['imported']
        );

        $cookie_list = '';
        if ( ! empty( $results['cookies'] ) ) {
            foreach ( $results['cookies'] as $c ) {
                $cookie_list .= "  • {$c['cookie_name']} ({$c['category']})\n";
            }
        }

        $message = sprintf(
            __( "Hi,\n\nThe scheduled cookie scan on %1\$s detected %2\$d new cookie(s):\n\n%3\$s\nThese cookies have been auto-imported. Please review their categories:\n%4\$s\n\n— RISE2 Cookie Consent", 'rise2-cookie-consent' ),
            $site_name,
            $results['imported'],
            $cookie_list,
            $settings_url
        );

        wp_mail( $admin_email, $subject, $message );
    }
}
