<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class R2CC_Frontend {

    public function __construct() {
        if ( ! Rise2_Cookie_Consent::get_setting( 'enabled', 1 ) ) {
            return;
        }

        add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
        add_action( 'wp_footer', array( $this, 'render_banner' ), 5 );
        add_action( 'wp_head', array( $this, 'output_inline_styles' ), 1 );
    }

    /**
     * Enqueue frontend scripts and styles
     */
    public function enqueue_assets() {
        wp_enqueue_style(
            'r2cc-frontend',
            R2CC_PLUGIN_URL . 'assets/css/frontend.css',
            array(),
            R2CC_VERSION
        );

        wp_enqueue_script(
            'r2cc-frontend',
            R2CC_PLUGIN_URL . 'assets/js/frontend.js',
            array(),
            R2CC_VERSION,
            true
        );

        // Pass settings to JS
        $categories = R2CC_Cookie_Categories::get_categories();
        $cookies_by_cat = array();
        foreach ( array_keys( $categories ) as $cat_slug ) {
            $cookies_by_cat[ $cat_slug ] = R2CC_Cookie_Database::get_cookies( $cat_slug );
        }

        $config = array(
            'ajaxUrl'          => admin_url( 'admin-ajax.php' ),
            'nonce'            => wp_create_nonce( 'r2cc_nonce' ),
            'consentExpiry'    => intval( Rise2_Cookie_Consent::get_setting( 'consent_expiry_days', 365 ) ),
            'consentVersion'   => Rise2_Cookie_Consent::get_setting( 'consent_version', '1.0' ),
            'position'         => Rise2_Cookie_Consent::get_setting( 'banner_position', 'bottom-left' ),
            'showRejectAll'    => intval( Rise2_Cookie_Consent::get_setting( 'show_reject_all', 1 ) ),
            'autoBlockScripts' => intval( Rise2_Cookie_Consent::get_setting( 'auto_block_scripts', 0 ) ),
            'categories'       => $categories,
            'cookiesByCat'     => $cookies_by_cat,
            'texts'            => array(
                'title'           => Rise2_Cookie_Consent::get_setting( 'banner_title' ),
                'description'     => Rise2_Cookie_Consent::get_setting( 'banner_description' ),
                'acceptAll'       => Rise2_Cookie_Consent::get_setting( 'btn_accept_label' ),
                'rejectAll'       => Rise2_Cookie_Consent::get_setting( 'btn_reject_label' ),
                'settings'        => Rise2_Cookie_Consent::get_setting( 'btn_settings_label' ),
                'save'            => Rise2_Cookie_Consent::get_setting( 'btn_save_label' ),
                'privacyUrl'      => Rise2_Cookie_Consent::get_setting( 'privacy_policy_url' ),
                'privacyLabel'    => Rise2_Cookie_Consent::get_setting( 'privacy_policy_label' ),
            ),
        );

        // Allow i18n and GCM to modify the config
        $config = apply_filters( 'r2cc_frontend_config', $config );

        wp_localize_script( 'r2cc-frontend', 'r2ccConfig', $config );
    }

    /**
     * Output inline CSS variables for customization
     */
    public function output_inline_styles() {
        $bg_color       = Rise2_Cookie_Consent::get_setting( 'banner_bg_color', '#1a1a2e' );
        $text_color     = Rise2_Cookie_Consent::get_setting( 'banner_text_color', '#e0e0e0' );
        $accept_bg      = Rise2_Cookie_Consent::get_setting( 'btn_accept_bg', '#4f46e5' );
        $accept_text    = Rise2_Cookie_Consent::get_setting( 'btn_accept_text', '#ffffff' );
        $reject_bg      = Rise2_Cookie_Consent::get_setting( 'btn_reject_bg', 'transparent' );
        $reject_text    = Rise2_Cookie_Consent::get_setting( 'btn_reject_text', '#e0e0e0' );
        $settings_bg    = Rise2_Cookie_Consent::get_setting( 'btn_settings_bg', 'transparent' );
        $settings_text  = Rise2_Cookie_Consent::get_setting( 'btn_settings_text', '#e0e0e0' );
        $custom_css     = Rise2_Cookie_Consent::get_setting( 'custom_css', '' );

        echo '<style id="r2cc-custom-vars">
:root {
    --r2cc-bg: ' . esc_attr( $bg_color ) . ';
    --r2cc-text: ' . esc_attr( $text_color ) . ';
    --r2cc-accept-bg: ' . esc_attr( $accept_bg ) . ';
    --r2cc-accept-text: ' . esc_attr( $accept_text ) . ';
    --r2cc-reject-bg: ' . esc_attr( $reject_bg ) . ';
    --r2cc-reject-text: ' . esc_attr( $reject_text ) . ';
    --r2cc-settings-bg: ' . esc_attr( $settings_bg ) . ';
    --r2cc-settings-text: ' . esc_attr( $settings_text ) . ';
}
</style>';

        if ( ! empty( $custom_css ) ) {
            echo '<style id="r2cc-custom-css">' . wp_strip_all_tags( $custom_css ) . '</style>';
        }
    }

    /**
     * Render the minimal banner HTML shell (content built by JS for performance)
     */
    public function render_banner() {
        $position = Rise2_Cookie_Consent::get_setting( 'banner_position', 'bottom-left' );
        $show_powered = Rise2_Cookie_Consent::get_setting( 'show_powered_by', 1 );
        ?>
        <!-- RISE2 Cookie Consent v<?php echo esc_attr( R2CC_VERSION ); ?> -->
        <div id="r2cc-banner" class="r2cc-banner r2cc-position-<?php echo esc_attr( $position ); ?>" role="dialog" aria-modal="false" aria-label="<?php esc_attr_e( 'Cookie Consent', 'rise2-cookie-consent' ); ?>" style="display:none;">
            <div class="r2cc-banner-inner">
                <div class="r2cc-banner-content">
                    <div class="r2cc-banner-header">
                        <svg class="r2cc-icon-cookie" width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                            <path d="M12 2C6.477 2 2 6.477 2 12s4.477 10 10 10 10-4.477 10-10c0-.34-.018-.676-.053-1.008a3.5 3.5 0 01-2.447-.96 3.5 3.5 0 01-.96-2.447A3.494 3.494 0 0116.585 5.5a3.5 3.5 0 01-.632-3.482A10.054 10.054 0 0012 2z" fill="currentColor" opacity="0.15"/>
                            <path d="M12 2C6.477 2 2 6.477 2 12s4.477 10 10 10 10-4.477 10-10c0-.34-.018-.676-.053-1.008a3.5 3.5 0 01-2.447-.96 3.5 3.5 0 01-.96-2.447A3.494 3.494 0 0116.585 5.5a3.5 3.5 0 01-.632-3.482A10.054 10.054 0 0012 2z" stroke="currentColor" stroke-width="1.5"/>
                            <circle cx="8.5" cy="10.5" r="1.5" fill="currentColor"/>
                            <circle cx="11" cy="15" r="1.5" fill="currentColor"/>
                            <circle cx="15" cy="11" r="1" fill="currentColor"/>
                            <circle cx="7" cy="14" r="1" fill="currentColor"/>
                        </svg>
                        <h3 class="r2cc-banner-title"></h3>
                    </div>
                    <p class="r2cc-banner-description"></p>
                    <div class="r2cc-banner-actions">
                        <button type="button" class="r2cc-btn r2cc-btn-settings" data-action="open-settings"></button>
                        <button type="button" class="r2cc-btn r2cc-btn-reject" data-action="reject-all" style="display:none;"></button>
                        <button type="button" class="r2cc-btn r2cc-btn-accept" data-action="accept-all"></button>
                    </div>
                    <div class="r2cc-privacy-link-wrap"></div>
                </div>
            </div>
        </div>

        <!-- Settings Panel -->
        <div id="r2cc-settings-panel" class="r2cc-settings-panel" role="dialog" aria-modal="true" aria-label="<?php esc_attr_e( 'Cookie Settings', 'rise2-cookie-consent' ); ?>" style="display:none;">
            <div class="r2cc-settings-overlay"></div>
            <div class="r2cc-settings-inner">
                <div class="r2cc-settings-header">
                    <h3 class="r2cc-settings-title"><?php esc_html_e( 'Cookie Settings', 'rise2-cookie-consent' ); ?></h3>
                    <button type="button" class="r2cc-settings-close" aria-label="<?php esc_attr_e( 'Close', 'rise2-cookie-consent' ); ?>">
                        <svg width="20" height="20" viewBox="0 0 20 20" fill="none"><path d="M15 5L5 15M5 5l10 10" stroke="currentColor" stroke-width="2" stroke-linecap="round"/></svg>
                    </button>
                </div>
                <div class="r2cc-settings-body">
                    <p class="r2cc-settings-desc"><?php esc_html_e( 'When you visit any website, it may store or retrieve information on your browser, mostly in the form of cookies. This information might be about you, your preferences, or your device and is mostly used to make the site work as you expect it to. The information does not usually directly identify you, but it can give you a more personalized web experience.', 'rise2-cookie-consent' ); ?></p>
                    <div class="r2cc-categories-list"></div>
                </div>
                <div class="r2cc-settings-footer">
                    <button type="button" class="r2cc-btn r2cc-btn-accept" data-action="accept-all"></button>
                    <button type="button" class="r2cc-btn r2cc-btn-reject" data-action="reject-all"></button>
                    <button type="button" class="r2cc-btn r2cc-btn-save" data-action="save-preferences"></button>
                </div>
                <?php if ( $show_powered ) : ?>
                <div class="r2cc-powered">
                    <?php printf(
                        /* translators: %s: link to RISE2 Studio */
                        esc_html__( 'Powered by %s', 'rise2-cookie-consent' ),
                        '<a href="https://rise2.studio" target="_blank" rel="noopener">RISE2 Cookie Consent</a>'
                    ); ?>
                </div>
                <?php endif; ?>
            </div>
        </div>

        <!-- Floating re-open button -->
        <button type="button" id="r2cc-reopen" class="r2cc-reopen r2cc-reopen-<?php echo esc_attr( $position ); ?>" style="display:none;" aria-label="<?php esc_attr_e( 'Cookie Settings', 'rise2-cookie-consent' ); ?>" title="<?php esc_attr_e( 'Cookie Settings', 'rise2-cookie-consent' ); ?>">
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 2C6.477 2 2 6.477 2 12s4.477 10 10 10 10-4.477 10-10c0-.34-.018-.676-.053-1.008a3.5 3.5 0 01-2.447-.96 3.5 3.5 0 01-.96-2.447A3.494 3.494 0 0116.585 5.5a3.5 3.5 0 01-.632-3.482A10.054 10.054 0 0012 2z" fill="currentColor" opacity="0.15"/>
                <path d="M12 2C6.477 2 2 6.477 2 12s4.477 10 10 10 10-4.477 10-10c0-.34-.018-.676-.053-1.008a3.5 3.5 0 01-2.447-.96 3.5 3.5 0 01-.96-2.447A3.494 3.494 0 0116.585 5.5a3.5 3.5 0 01-.632-3.482A10.054 10.054 0 0012 2z" stroke="currentColor" stroke-width="1.5"/>
                <circle cx="8.5" cy="10.5" r="1.5" fill="currentColor"/>
                <circle cx="11" cy="15" r="1.5" fill="currentColor"/>
                <circle cx="15" cy="11" r="1" fill="currentColor"/>
            </svg>
        </button>
        <!-- /RISE2 Cookie Consent -->
        <?php
    }
}

new R2CC_Frontend();
