<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Google Consent Mode v2 Integration
 *
 * Implements the full GCM v2 spec:
 * - Outputs gtag('consent','default',{...}) as early as possible in <head>
 * - Updates consent state via gtag('consent','update',{...}) when user interacts
 * - Maps our cookie categories to Google's consent types
 * - Supports wait_for_update for async consent tools
 * - Supports ads_data_redaction and url_passthrough
 *
 * @see https://developers.google.com/tag-platform/security/guides/consent
 */
class R2CC_Google_Consent_Mode {

    /**
     * Mapping: our categories → Google consent types
     *
     * Google Consent Mode v2 types:
     *   ad_storage           - Enables storage for advertising (cookies, etc.)
     *   ad_user_data         - Consent to send user data to Google for advertising
     *   ad_personalization   - Consent to personalized advertising
     *   analytics_storage    - Enables storage for analytics (cookies, etc.)
     *   functionality_storage - Enables storage for functional features
     *   personalization_storage - Enables storage for personalization
     *   security_storage     - Enables storage for security (always granted)
     */
    const CATEGORY_MAP = array(
        'necessary'  => array( 'security_storage' ),
        'functional' => array( 'functionality_storage', 'personalization_storage' ),
        'analytics'  => array( 'analytics_storage' ),
        'marketing'  => array( 'ad_storage', 'ad_user_data', 'ad_personalization' ),
    );

    /**
     * All Google consent types for reference
     */
    const ALL_CONSENT_TYPES = array(
        'ad_storage',
        'ad_user_data',
        'ad_personalization',
        'analytics_storage',
        'functionality_storage',
        'personalization_storage',
        'security_storage',
    );

    public function __construct() {
        if ( ! Rise2_Cookie_Consent::get_setting( 'gcm_enabled', 0 ) ) {
            return;
        }

        // Must fire BEFORE any gtag/GTM scripts — priority 1 in wp_head
        add_action( 'wp_head', array( $this, 'output_default_consent' ), 1 );

        // Pass GCM config to frontend JS
        add_filter( 'r2cc_frontend_config', array( $this, 'add_gcm_config' ) );
    }

    /**
     * Output the default consent state as early as possible in <head>.
     * This MUST appear before the Google Tag (gtag.js) or GTM snippet.
     */
    public function output_default_consent() {
        $wait_for_update   = intval( Rise2_Cookie_Consent::get_setting( 'gcm_wait_for_update', 500 ) );
        $ads_data_redaction = intval( Rise2_Cookie_Consent::get_setting( 'gcm_ads_data_redaction', 1 ) );
        $url_passthrough    = intval( Rise2_Cookie_Consent::get_setting( 'gcm_url_passthrough', 0 ) );
        $region             = Rise2_Cookie_Consent::get_setting( 'gcm_region', '' );

        // Build default state from existing consent cookie (if present) or deny all
        $default_state = $this->get_default_state();

        // Region array for conditional consent (empty = worldwide)
        $region_js = '';
        if ( ! empty( $region ) ) {
            $regions = array_map( 'trim', explode( ',', $region ) );
            $regions = array_map( 'strtoupper', $regions );
            $region_js = "region: " . wp_json_encode( $regions ) . ",";
        }
        ?>
<script data-r2cc-gcm="default">
window.dataLayer = window.dataLayer || [];
function gtag(){dataLayer.push(arguments);}

// Default consent state — deny everything except security_storage
gtag('consent', 'default', {
    <?php echo $region_js; ?>

    'ad_storage': '<?php echo esc_js( $default_state['ad_storage'] ); ?>',
    'ad_user_data': '<?php echo esc_js( $default_state['ad_user_data'] ); ?>',
    'ad_personalization': '<?php echo esc_js( $default_state['ad_personalization'] ); ?>',
    'analytics_storage': '<?php echo esc_js( $default_state['analytics_storage'] ); ?>',
    'functionality_storage': '<?php echo esc_js( $default_state['functionality_storage'] ); ?>',
    'personalization_storage': '<?php echo esc_js( $default_state['personalization_storage'] ); ?>',
    'security_storage': 'granted',
    'wait_for_update': <?php echo intval( $wait_for_update ); ?>

});

<?php if ( $ads_data_redaction ) : ?>
gtag('set', 'ads_data_redaction', true);
<?php endif; ?>

<?php if ( $url_passthrough ) : ?>
gtag('set', 'url_passthrough', true);
<?php endif; ?>
</script>
        <?php
    }

    /**
     * Get the default consent state.
     * If user already has a consent cookie, use that. Otherwise deny all.
     */
    private function get_default_state() {
        $state = array();

        foreach ( self::ALL_CONSENT_TYPES as $type ) {
            $state[ $type ] = 'denied';
        }

        // Security storage is always granted
        $state['security_storage'] = 'granted';

        // Check for existing consent cookie (server-side check for returning visitors)
        if ( isset( $_COOKIE['r2cc_consent'] ) ) {
            $consent = json_decode( stripslashes( $_COOKIE['r2cc_consent'] ), true );
            if ( $consent && isset( $consent['categories'] ) ) {
                foreach ( $consent['categories'] as $category ) {
                    if ( isset( self::CATEGORY_MAP[ $category ] ) ) {
                        foreach ( self::CATEGORY_MAP[ $category ] as $gcm_type ) {
                            $state[ $gcm_type ] = 'granted';
                        }
                    }
                }
            }
        }

        return $state;
    }

    /**
     * Add GCM configuration to the frontend JS config object
     */
    public function add_gcm_config( $config ) {
        $config['gcm'] = array(
            'enabled'     => true,
            'categoryMap' => self::CATEGORY_MAP,
        );
        return $config;
    }

    /**
     * Get the mapping for use in JS (static helper)
     */
    public static function get_category_map() {
        return self::CATEGORY_MAP;
    }
}
