<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Internationalization & Translation Support
 *
 * Provides:
 * - Built-in translations (starting with Croatian)
 * - Automatic language detection (WP locale, WPML, Polylang)
 * - WPML String Translation integration
 * - Runtime frontend string translation via JS config
 * - Translatable category names and descriptions
 */
class R2CC_I18n {

    /**
     * Current language code
     */
    private static $current_lang = null;

    /**
     * Built-in translations
     */
    private static $translations = array();

    /**
     * Initialize i18n support
     */
    public static function init() {
        self::$translations = self::get_builtin_translations();

        // Register strings with WPML String Translation (if available)
        add_action( 'init', array( __CLASS__, 'register_wpml_strings' ), 20 );

        // Hook into frontend config to provide translated texts
        add_filter( 'r2cc_frontend_config', array( __CLASS__, 'localize_frontend_config' ) );

        // Register WPML language switcher support
        add_filter( 'r2cc_banner_texts', array( __CLASS__, 'translate_banner_texts' ) );
    }

    /**
     * Get current language code
     */
    public static function get_current_language() {
        if ( self::$current_lang !== null ) {
            return self::$current_lang;
        }

        // 1. WPML
        if ( defined( 'ICL_LANGUAGE_CODE' ) ) {
            self::$current_lang = ICL_LANGUAGE_CODE;
            return self::$current_lang;
        }

        // 2. Polylang
        if ( function_exists( 'pll_current_language' ) ) {
            self::$current_lang = pll_current_language( 'slug' );
            return self::$current_lang;
        }

        // 3. TranslatePress
        if ( class_exists( 'TRP_Translate_Press' ) ) {
            global $TRP_LANGUAGE;
            if ( ! empty( $TRP_LANGUAGE ) ) {
                self::$current_lang = substr( $TRP_LANGUAGE, 0, 2 );
                return self::$current_lang;
            }
        }

        // 4. WordPress locale fallback
        $locale = get_locale();
        self::$current_lang = substr( $locale, 0, 2 );

        return self::$current_lang;
    }

    /**
     * Translate a string with fallback chain:
     * WPML String Translation → Built-in translations → Original (English)
     */
    public static function translate( $string, $context = 'banner' ) {
        // 1. Try WPML String Translation
        if ( function_exists( 'icl_t' ) ) {
            $translated = icl_t( 'rise2-cookie-consent', $context . ' - ' . $string, $string );
            if ( $translated !== $string ) {
                return $translated;
            }
        }

        // 2. Try built-in translations
        $lang = self::get_current_language();
        if ( isset( self::$translations[ $lang ][ $string ] ) ) {
            return self::$translations[ $lang ][ $string ];
        }

        // 3. Return original
        return $string;
    }

    /**
     * Register translatable strings with WPML String Translation
     */
    public static function register_wpml_strings() {
        if ( ! function_exists( 'icl_register_string' ) ) {
            return;
        }

        $strings = array(
            'banner_title'       => get_option( 'r2cc_banner_title', 'We value your privacy' ),
            'banner_description' => get_option( 'r2cc_banner_description', '' ),
            'btn_accept_label'   => get_option( 'r2cc_btn_accept_label', 'Accept All' ),
            'btn_reject_label'   => get_option( 'r2cc_btn_reject_label', 'Reject All' ),
            'btn_settings_label' => get_option( 'r2cc_btn_settings_label', 'Cookie Settings' ),
            'btn_save_label'     => get_option( 'r2cc_btn_save_label', 'Save Preferences' ),
            'privacy_policy_label' => get_option( 'r2cc_privacy_policy_label', 'Privacy Policy' ),
        );

        foreach ( $strings as $name => $value ) {
            icl_register_string( 'rise2-cookie-consent', 'banner - ' . $name, $value );
        }

        // Register category names and descriptions
        $categories = R2CC_Cookie_Categories::get_categories();
        foreach ( $categories as $slug => $cat ) {
            icl_register_string( 'rise2-cookie-consent', 'category_label - ' . $slug, $cat['label'] );
            icl_register_string( 'rise2-cookie-consent', 'category_desc - ' . $slug, $cat['description'] );
        }
    }

    /**
     * Translate the banner texts for the frontend
     */
    public static function translate_banner_texts( $texts ) {
        $lang = self::get_current_language();

        // If not English and we have built-in translations, apply them as defaults
        // WPML translated values from the DB take priority
        if ( $lang !== 'en' && isset( self::$translations[ $lang ] ) ) {
            $trans = self::$translations[ $lang ];

            foreach ( $texts as $key => $value ) {
                // Check if WPML has a translation first
                if ( function_exists( 'icl_t' ) ) {
                    $wpml_val = icl_t( 'rise2-cookie-consent', 'banner - ' . $key, $value );
                    if ( $wpml_val !== $value ) {
                        $texts[ $key ] = $wpml_val;
                        continue;
                    }
                }

                // Fall back to built-in
                if ( isset( $trans[ $value ] ) ) {
                    $texts[ $key ] = $trans[ $value ];
                }
            }
        }

        return $texts;
    }

    /**
     * Localize the frontend config with translated categories
     */
    public static function localize_frontend_config( $config ) {
        $lang = self::get_current_language();

        // Translate category labels and descriptions
        if ( isset( $config['categories'] ) ) {
            foreach ( $config['categories'] as $slug => &$cat ) {
                // Try WPML first
                if ( function_exists( 'icl_t' ) ) {
                    $cat['label'] = icl_t( 'rise2-cookie-consent', 'category_label - ' . $slug, $cat['label'] );
                    $cat['description'] = icl_t( 'rise2-cookie-consent', 'category_desc - ' . $slug, $cat['description'] );
                }
                // Then try built-in
                elseif ( $lang !== 'en' && isset( self::$translations[ $lang ] ) ) {
                    $trans = self::$translations[ $lang ];
                    if ( isset( $trans[ $cat['label'] ] ) ) {
                        $cat['label'] = $trans[ $cat['label'] ];
                    }
                    if ( isset( $trans[ $cat['description'] ] ) ) {
                        $cat['description'] = $trans[ $cat['description'] ];
                    }
                }
            }
            unset( $cat );
        }

        // Translate texts
        if ( isset( $config['texts'] ) ) {
            $config['texts'] = apply_filters( 'r2cc_banner_texts', $config['texts'] );
        }

        // Add current lang for JS
        $config['lang'] = $lang;

        // Add translated settings panel strings
        $config['settingsTexts'] = self::get_settings_panel_texts( $lang );

        return $config;
    }

    /**
     * Get translated settings panel static strings
     */
    private static function get_settings_panel_texts( $lang ) {
        $defaults = array(
            'settingsTitle'  => __( 'Cookie Settings', 'rise2-cookie-consent' ),
            'settingsDesc'   => __( 'When you visit any website, it may store or retrieve information on your browser, mostly in the form of cookies. This information might be about you, your preferences, or your device and is mostly used to make the site work as you expect it to. The information does not usually directly identify you, but it can give you a more personalized web experience.', 'rise2-cookie-consent' ),
            'alwaysActive'   => __( 'Always Active', 'rise2-cookie-consent' ),
            'cookieHeader'   => __( 'Cookie', 'rise2-cookie-consent' ),
            'durationHeader' => __( 'Duration', 'rise2-cookie-consent' ),
            'descHeader'     => __( 'Description', 'rise2-cookie-consent' ),
            'noCookies'      => __( 'No cookies registered in this category yet.', 'rise2-cookie-consent' ),
            'poweredBy'      => __( 'Powered by', 'rise2-cookie-consent' ),
        );

        if ( $lang !== 'en' && isset( self::$translations[ $lang ] ) ) {
            $trans = self::$translations[ $lang ];
            foreach ( $defaults as $key => $value ) {
                if ( isset( $trans[ $value ] ) ) {
                    $defaults[ $key ] = $trans[ $value ];
                }
            }
        }

        return $defaults;
    }

    /**
     * Get all built-in translations
     * Each language returns a map of English string → Translated string
     */
    private static function get_builtin_translations() {
        return array(
            'hr' => array(
                // Banner texts
                'We value your privacy' => 'Cijenimo vašu privatnost',
                'We use cookies to enhance your browsing experience, serve personalized content, and analyze our traffic. By clicking "Accept All", you consent to our use of cookies. You can customize your preferences by clicking "Cookie Settings".' =>
                    'Koristimo kolačiće za poboljšanje vašeg iskustva pregledavanja, pružanje personaliziranog sadržaja i analizu prometa. Klikom na "Prihvati sve" pristajete na korištenje kolačića. Svoje postavke možete prilagoditi klikom na "Postavke kolačića".',
                'Accept All'       => 'Prihvati sve',
                'Reject All'       => 'Odbij sve',
                'Cookie Settings'  => 'Postavke kolačića',
                'Save Preferences' => 'Spremi postavke',
                'Privacy Policy'   => 'Politika privatnosti',

                // Category labels
                'Strictly Necessary'      => 'Strogo neophodni',
                'Functional'              => 'Funkcionalni',
                'Analytics & Performance' => 'Analitički',
                'Marketing & Advertising' => 'Marketinški',

                // Category descriptions
                'These cookies are essential for the website to function properly. They enable basic features like page navigation, secure areas access, and shopping cart functionality. The website cannot function properly without these cookies, and they are always enabled.' =>
                    'Ovi kolačići neophodni su za pravilno funkcioniranje web stranice. Omogućuju osnovne značajke poput navigacije stranicom, pristupa sigurnim područjima i funkcionalnosti košarice za kupnju. Web stranica ne može ispravno funkcionirati bez ovih kolačića i uvijek su aktivni.',

                'Functional cookies enable the website to provide enhanced functionality and personalization. They may be set by us or by third-party providers whose services we have added to our pages. If you do not allow these cookies, some or all of these services may not function properly.' =>
                    'Funkcionalni kolačići omogućuju web stranici pružanje poboljšane funkcionalnosti i personalizacije. Mogu ih postaviti naši sustavi ili pružatelji usluga trećih strana čije smo usluge dodali na naše stranice. Ako ne dopustite ove kolačiće, neke ili sve ove usluge možda neće ispravno funkcionirati.',

                'These cookies allow us to count visits and traffic sources so we can measure and improve the performance of our site. They help us understand which pages are the most and least popular and see how visitors move around the site. All information these cookies collect is aggregated and therefore anonymous.' =>
                    'Ovi kolačići omogućuju nam brojanje posjeta i izvora prometa kako bismo mogli mjeriti i poboljšati performanse naše stranice. Pomažu nam razumjeti koje su stranice najpopularnije, a koje najmanje, te kako se posjetitelji kreću po stranici. Svi podaci koje ovi kolačići prikupljaju agregirani su i stoga anonimni.',

                'These cookies may be set through our site by our advertising partners. They may be used by those companies to build a profile of your interests and show you relevant advertisements on other sites. They do not directly store personal information but uniquely identify your browser and internet device.' =>
                    'Ove kolačiće mogu postaviti naši oglašivački partneri putem naše stranice. Tvrtke ih mogu koristiti za izradu profila vaših interesa i prikazivanje relevantnih oglasa na drugim stranicama. Ne pohranjuju izravno osobne podatke, već jedinstveno identificiraju vaš preglednik i internetski uređaj.',

                // Settings panel strings
                'When you visit any website, it may store or retrieve information on your browser, mostly in the form of cookies. This information might be about you, your preferences, or your device and is mostly used to make the site work as you expect it to. The information does not usually directly identify you, but it can give you a more personalized web experience.' =>
                    'Kada posjetite bilo koju web stranicu, ona može pohraniti ili dohvatiti informacije u vašem pregledniku, uglavnom u obliku kolačića. Te informacije mogu biti o vama, vašim preferencijama ili vašem uređaju i uglavnom se koriste kako bi stranica radila onako kako očekujete. Informacije vas obično ne identificiraju izravno, ali vam mogu pružiti personaliziranije web iskustvo.',

                'Always Active' => 'Uvijek aktivno',
                'Cookie'        => 'Kolačić',
                'Duration'      => 'Trajanje',
                'Description'   => 'Opis',
                'No cookies registered in this category yet.' => 'Još nema registriranih kolačića u ovoj kategoriji.',
                'Powered by'    => 'Pokreće',
                'Cookie Consent' => 'Privola za kolačiće',
                'Close'          => 'Zatvori',
            ),
        );
    }

    /**
     * Get list of available built-in languages
     */
    public static function get_available_languages() {
        return array(
            'en' => 'English',
            'hr' => 'Hrvatski',
        );
    }

    /**
     * Check if a language has built-in translation
     */
    public static function has_builtin_translation( $lang ) {
        return isset( self::$translations[ $lang ] );
    }
}
