<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Privacy Policy Page Generator
 *
 * Generates a comprehensive GDPR-compliant privacy policy page.
 * Supports English and Croatian, auto-populates with site data,
 * and can be inserted as a WP page or copied as HTML.
 */
class R2CC_Privacy_Policy_Generator {

    /**
     * Generate the privacy policy content
     *
     * @param string $lang Language code ('en' or 'hr')
     * @param array  $data Override data (site_name, site_url, contact_email, etc.)
     * @return string HTML content
     */
    public static function generate( $lang = 'en', $data = array() ) {
        $defaults = array(
            'site_name'      => get_bloginfo( 'name' ),
            'site_url'       => home_url(),
            'contact_email'  => get_option( 'admin_email' ),
            'company_name'   => get_option( 'r2cc_pp_company_name', get_bloginfo( 'name' ) ),
            'company_address' => get_option( 'r2cc_pp_company_address', '' ),
            'company_oib'    => get_option( 'r2cc_pp_company_oib', '' ),
            'dpo_email'      => get_option( 'r2cc_pp_dpo_email', get_option( 'admin_email' ) ),
            'last_updated'   => current_time( 'Y-m-d' ),
        );

        $d = wp_parse_args( $data, $defaults );

        // Get cookies from DB grouped by category
        $categories = R2CC_Cookie_Categories::get_categories();
        $cookies_by_cat = array();
        foreach ( array_keys( $categories ) as $cat ) {
            $cookies_by_cat[ $cat ] = R2CC_Cookie_Database::get_cookies( $cat );
        }

        if ( $lang === 'hr' ) {
            return self::template_hr( $d, $categories, $cookies_by_cat );
        }

        return self::template_en( $d, $categories, $cookies_by_cat );
    }

    /**
     * Create a WordPress page with the policy content
     */
    public static function create_page( $lang = 'en', $data = array() ) {
        $content = self::generate( $lang, $data );

        $title = $lang === 'hr' ? 'Politika privatnosti' : 'Privacy Policy';

        // Check if page already exists
        $existing = get_page_by_title( $title, OBJECT, 'page' );
        if ( $existing ) {
            wp_update_post( array(
                'ID'           => $existing->ID,
                'post_content' => $content,
            ) );
            return $existing->ID;
        }

        $page_id = wp_insert_post( array(
            'post_title'   => $title,
            'post_content' => $content,
            'post_status'  => 'draft',
            'post_type'    => 'page',
            'post_author'  => get_current_user_id(),
        ) );

        if ( $page_id && ! is_wp_error( $page_id ) ) {
            // Set as WP privacy policy page
            update_option( 'wp_page_for_privacy_policy', $page_id );

            // Auto-set our privacy URL setting
            update_option( 'r2cc_privacy_policy_url', get_permalink( $page_id ) );
        }

        return $page_id;
    }

    /**
     * Build the cookie table HTML for a given category
     */
    private static function cookie_table( $cookies, $lang = 'en' ) {
        if ( empty( $cookies ) ) {
            return $lang === 'hr'
                ? '<p><em>Trenutno nema registriranih kolačića u ovoj kategoriji.</em></p>'
                : '<p><em>No cookies currently registered in this category.</em></p>';
        }

        $header_name = $lang === 'hr' ? 'Naziv kolačića' : 'Cookie Name';
        $header_dur  = $lang === 'hr' ? 'Trajanje' : 'Duration';
        $header_desc = $lang === 'hr' ? 'Opis' : 'Description';

        $html = '<table style="width:100%;border-collapse:collapse;margin:12px 0 20px;">
            <thead>
                <tr style="background:#f5f5f5;text-align:left;">
                    <th style="padding:8px 12px;border:1px solid #ddd;font-size:13px;">' . $header_name . '</th>
                    <th style="padding:8px 12px;border:1px solid #ddd;font-size:13px;">' . $header_dur . '</th>
                    <th style="padding:8px 12px;border:1px solid #ddd;font-size:13px;">' . $header_desc . '</th>
                </tr>
            </thead>
            <tbody>';

        foreach ( $cookies as $c ) {
            $html .= '<tr>
                <td style="padding:8px 12px;border:1px solid #ddd;font-family:monospace;font-size:12px;">' . esc_html( $c->cookie_name ) . '</td>
                <td style="padding:8px 12px;border:1px solid #ddd;font-size:13px;">' . esc_html( $c->duration ?: '—' ) . '</td>
                <td style="padding:8px 12px;border:1px solid #ddd;font-size:13px;">' . esc_html( $c->description ?: '—' ) . '</td>
            </tr>';
        }

        $html .= '</tbody></table>';
        return $html;
    }

    /**
     * English Template
     */
    private static function template_en( $d, $categories, $cookies_by_cat ) {
        $site       = esc_html( $d['site_name'] );
        $url        = esc_url( $d['site_url'] );
        $company    = esc_html( $d['company_name'] );
        $address    = esc_html( $d['company_address'] );
        $oib        = esc_html( $d['company_oib'] );
        $email      = esc_html( $d['contact_email'] );
        $dpo        = esc_html( $d['dpo_email'] );
        $updated    = esc_html( $d['last_updated'] );

        $company_info = $company;
        if ( $address ) $company_info .= ', ' . $address;
        if ( $oib ) $company_info .= ' (OIB/ID: ' . $oib . ')';

        ob_start();
        ?>
<!-- Privacy Policy generated by RISE2 Cookie Consent -->
<p><strong>Last updated:</strong> <?php echo $updated; ?></p>

<h2>1. Introduction</h2>
<p><?php echo $company; ?> ("we", "us", or "our") operates the website <a href="<?php echo $url; ?>"><?php echo $site; ?></a>. This Privacy Policy explains how we collect, use, disclose, and protect your personal data when you visit our website, in accordance with the General Data Protection Regulation (GDPR – EU Regulation 2016/679) and applicable national legislation.</p>

<p>By using our website, you acknowledge that you have read and understood this Privacy Policy. If you do not agree with our practices, please do not use our website.</p>

<h2>2. Data Controller</h2>
<p>The data controller responsible for your personal data is:</p>
<p><strong><?php echo $company_info; ?></strong><br>
Email: <a href="mailto:<?php echo $email; ?>"><?php echo $email; ?></a></p>
<?php if ( $dpo && $dpo !== $email ) : ?>
<p>Data Protection Officer (DPO): <a href="mailto:<?php echo $dpo; ?>"><?php echo $dpo; ?></a></p>
<?php endif; ?>

<h2>3. What Personal Data We Collect</h2>
<p>We may collect and process the following categories of personal data:</p>
<ul>
    <li><strong>Identity data:</strong> name, username, or similar identifier</li>
    <li><strong>Contact data:</strong> email address, telephone number, postal address</li>
    <li><strong>Technical data:</strong> IP address (anonymized), browser type and version, operating system, time zone, language preferences</li>
    <li><strong>Usage data:</strong> pages visited, time spent on pages, click patterns, referral source</li>
    <li><strong>Cookie data:</strong> information collected through cookies and similar technologies (see Section 7 below)</li>
    <li><strong>Communication data:</strong> any data you provide when contacting us via forms, email, or other channels</li>
</ul>

<h2>4. How We Collect Your Data</h2>
<p>We collect personal data through:</p>
<ul>
    <li><strong>Direct interactions:</strong> when you fill in forms, subscribe to newsletters, create an account, make a purchase, or contact us</li>
    <li><strong>Automated technologies:</strong> as you navigate our website, we may automatically collect technical and usage data through cookies, server logs, and similar technologies</li>
    <li><strong>Third parties:</strong> we may receive data from analytics providers, advertising networks, and social media platforms</li>
</ul>

<h2>5. Legal Basis for Processing</h2>
<p>We process your personal data based on one or more of the following legal grounds under Article 6(1) of the GDPR:</p>
<ul>
    <li><strong>Consent (Art. 6(1)(a)):</strong> you have given clear consent for us to process your personal data for a specific purpose (e.g., marketing cookies, newsletters)</li>
    <li><strong>Contract (Art. 6(1)(b)):</strong> processing is necessary for the performance of a contract with you or to take steps at your request before entering a contract</li>
    <li><strong>Legal obligation (Art. 6(1)(c)):</strong> processing is necessary to comply with a legal obligation</li>
    <li><strong>Legitimate interest (Art. 6(1)(f)):</strong> processing is necessary for our legitimate interests or those of a third party, provided your rights do not override those interests</li>
</ul>

<h2>6. How We Use Your Data</h2>
<p>We use the collected data for the following purposes:</p>
<ul>
    <li>To provide, operate, and maintain our website</li>
    <li>To improve and personalize your experience</li>
    <li>To understand and analyze how you use our website</li>
    <li>To communicate with you, including for customer service, updates, and marketing (with your consent)</li>
    <li>To process transactions and manage orders</li>
    <li>To comply with legal obligations</li>
    <li>To detect, prevent, and address technical issues or fraud</li>
</ul>

<h2>7. Cookies and Tracking Technologies</h2>
<p>Our website uses cookies and similar tracking technologies. A cookie is a small text file placed on your device when you visit a website. We use the following categories of cookies:</p>

<h3>7.1 Strictly Necessary Cookies</h3>
<p><?php echo esc_html( $categories['necessary']['description'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['necessary'] ?? array(), 'en' ); ?>

<h3>7.2 Functional Cookies</h3>
<p><?php echo esc_html( $categories['functional']['description'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['functional'] ?? array(), 'en' ); ?>

<h3>7.3 Analytics & Performance Cookies</h3>
<p><?php echo esc_html( $categories['analytics']['description'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['analytics'] ?? array(), 'en' ); ?>

<h3>7.4 Marketing & Advertising Cookies</h3>
<p><?php echo esc_html( $categories['marketing']['description'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['marketing'] ?? array(), 'en' ); ?>

<h3>7.5 Managing Your Cookie Preferences</h3>
<p>When you first visit our website, you will be presented with a cookie consent banner that allows you to accept or reject cookie categories. You can change your preferences at any time by clicking the cookie icon in the bottom corner of the page, or by using the link below:</p>
<p><a href="#" onclick="if(window.R2CookieConsent){R2CookieConsent.showSettings();}return false;">Manage Cookie Preferences</a></p>
<p>You can also control cookies through your browser settings. Please note that disabling certain cookies may affect the functionality of our website.</p>

<h2>8. Data Sharing and Third Parties</h2>
<p>We may share your personal data with the following categories of recipients:</p>
<ul>
    <li><strong>Service providers:</strong> hosting, analytics, email marketing, payment processing</li>
    <li><strong>Advertising partners:</strong> only with your explicit consent for marketing cookies</li>
    <li><strong>Legal authorities:</strong> when required by law or to protect our rights</li>
</ul>
<p>We do not sell your personal data to third parties.</p>

<h2>9. International Data Transfers</h2>
<p>Some of our third-party service providers may be located outside the European Economic Area (EEA). When transferring data outside the EEA, we ensure appropriate safeguards are in place, such as Standard Contractual Clauses (SCCs) approved by the European Commission, or transfers to countries with an adequate level of data protection.</p>

<h2>10. Data Retention</h2>
<p>We retain your personal data only for as long as necessary to fulfill the purposes for which it was collected, including satisfying legal, accounting, or reporting requirements. Cookie consent records are retained for the duration of the consent period, after which they are automatically deleted.</p>

<h2>11. Your Rights Under GDPR</h2>
<p>Under the GDPR, you have the following rights regarding your personal data:</p>
<ul>
    <li><strong>Right of access (Art. 15):</strong> request a copy of your personal data</li>
    <li><strong>Right to rectification (Art. 16):</strong> request correction of inaccurate data</li>
    <li><strong>Right to erasure (Art. 17):</strong> request deletion of your data ("right to be forgotten")</li>
    <li><strong>Right to restrict processing (Art. 18):</strong> request limitation of data processing</li>
    <li><strong>Right to data portability (Art. 20):</strong> receive your data in a structured, machine-readable format</li>
    <li><strong>Right to object (Art. 21):</strong> object to processing based on legitimate interest or direct marketing</li>
    <li><strong>Right to withdraw consent (Art. 7):</strong> withdraw your consent at any time without affecting prior processing</li>
</ul>
<p>To exercise any of these rights, please contact us at: <a href="mailto:<?php echo $dpo; ?>"><?php echo $dpo; ?></a></p>
<p>You also have the right to lodge a complaint with your national data protection authority.</p>

<h2>12. Data Security</h2>
<p>We implement appropriate technical and organizational measures to protect your personal data against unauthorized access, alteration, disclosure, or destruction. These measures include encryption, access controls, and regular security assessments.</p>

<h2>13. Children's Privacy</h2>
<p>Our website is not intended for children under the age of 16. We do not knowingly collect personal data from children. If we become aware that we have collected data from a child under 16 without parental consent, we will take steps to delete it.</p>

<h2>14. Changes to This Policy</h2>
<p>We may update this Privacy Policy from time to time. Any changes will be posted on this page with an updated revision date. We encourage you to review this page periodically.</p>

<h2>15. Contact Us</h2>
<p>If you have any questions about this Privacy Policy or how we handle your personal data, please contact us:</p>
<p><strong><?php echo $company; ?></strong><br>
Email: <a href="mailto:<?php echo $email; ?>"><?php echo $email; ?></a><br>
Website: <a href="<?php echo $url; ?>"><?php echo $site; ?></a></p>
<?php
        return ob_get_clean();
    }

    /**
     * Croatian Template
     */
    private static function template_hr( $d, $categories, $cookies_by_cat ) {
        $site       = esc_html( $d['site_name'] );
        $url        = esc_url( $d['site_url'] );
        $company    = esc_html( $d['company_name'] );
        $address    = esc_html( $d['company_address'] );
        $oib        = esc_html( $d['company_oib'] );
        $email      = esc_html( $d['contact_email'] );
        $dpo        = esc_html( $d['dpo_email'] );
        $updated    = esc_html( $d['last_updated'] );

        $company_info = $company;
        if ( $address ) $company_info .= ', ' . $address;
        if ( $oib ) $company_info .= ' (OIB: ' . $oib . ')';

        // Croatian category descriptions
        $cat_desc_hr = array(
            'necessary'  => 'Ovi kolačići neophodni su za pravilno funkcioniranje web stranice. Omogućuju osnovne značajke poput navigacije stranicom, pristupa sigurnim područjima i funkcionalnosti košarice za kupnju. Web stranica ne može ispravno funkcionirati bez ovih kolačića i uvijek su aktivni.',
            'functional' => 'Funkcionalni kolačići omogućuju web stranici pružanje poboljšane funkcionalnosti i personalizacije. Mogu ih postaviti naši sustavi ili pružatelji usluga trećih strana čije smo usluge dodali na naše stranice. Ako ne dopustite ove kolačiće, neke ili sve ove usluge možda neće ispravno funkcionirati.',
            'analytics'  => 'Ovi kolačići omogućuju nam brojanje posjeta i izvora prometa kako bismo mogli mjeriti i poboljšati performanse naše stranice. Pomažu nam razumjeti koje su stranice najpopularnije, a koje najmanje, te kako se posjetitelji kreću po stranici. Svi podaci koje ovi kolačići prikupljaju agregirani su i stoga anonimni.',
            'marketing'  => 'Ove kolačiće mogu postaviti naši oglašivački partneri putem naše stranice. Tvrtke ih mogu koristiti za izradu profila vaših interesa i prikazivanje relevantnih oglasa na drugim stranicama. Ne pohranjuju izravno osobne podatke, već jedinstveno identificiraju vaš preglednik i internetski uređaj.',
        );

        ob_start();
        ?>
<!-- Politika privatnosti generirana putem RISE2 Cookie Consent -->
<p><strong>Zadnje ažuriranje:</strong> <?php echo $updated; ?></p>

<h2>1. Uvod</h2>
<p><?php echo $company; ?> ("mi", "nas" ili "naš") upravlja web stranicom <a href="<?php echo $url; ?>"><?php echo $site; ?></a>. Ova Politika privatnosti objašnjava kako prikupljamo, koristimo, otkrivamo i štitimo vaše osobne podatke kada posjećujete našu web stranicu, u skladu s Općom uredbom o zaštiti podataka (GDPR – Uredba EU 2016/679), Zakonom o provedbi Opće uredbe o zaštiti podataka (NN 42/18) te ostalim primjenjivim propisima Republike Hrvatske.</p>

<p>Korištenjem naše web stranice potvrđujete da ste pročitali i razumjeli ovu Politiku privatnosti. Ako se ne slažete s našim praksama, molimo vas da ne koristite našu web stranicu.</p>

<h2>2. Voditelj obrade</h2>
<p>Voditelj obrade odgovoran za vaše osobne podatke je:</p>
<p><strong><?php echo $company_info; ?></strong><br>
E-pošta: <a href="mailto:<?php echo $email; ?>"><?php echo $email; ?></a></p>
<?php if ( $dpo && $dpo !== $email ) : ?>
<p>Službenik za zaštitu podataka (DPO): <a href="mailto:<?php echo $dpo; ?>"><?php echo $dpo; ?></a></p>
<?php endif; ?>

<h2>3. Koje osobne podatke prikupljamo</h2>
<p>Možemo prikupljati i obrađivati sljedeće kategorije osobnih podataka:</p>
<ul>
    <li><strong>Identifikacijski podaci:</strong> ime, korisničko ime ili sličan identifikator</li>
    <li><strong>Kontaktni podaci:</strong> adresa e-pošte, telefonski broj, poštanska adresa</li>
    <li><strong>Tehnički podaci:</strong> IP adresa (anonimizirana), vrsta i verzija preglednika, operacijski sustav, vremenska zona, jezične postavke</li>
    <li><strong>Podaci o korištenju:</strong> posjećene stranice, vrijeme provedeno na stranicama, obrasci klikanja, izvor posjeta</li>
    <li><strong>Podaci o kolačićima:</strong> informacije prikupljene putem kolačića i sličnih tehnologija (vidjeti odjeljak 7. u nastavku)</li>
    <li><strong>Komunikacijski podaci:</strong> svi podaci koje nam dostavite kada nas kontaktirate putem obrazaca, e-pošte ili drugih kanala</li>
</ul>

<h2>4. Kako prikupljamo vaše podatke</h2>
<p>Osobne podatke prikupljamo putem:</p>
<ul>
    <li><strong>Izravne interakcije:</strong> kada ispunjavate obrasce, pretplaćujete se na newslettere, kreirate račun, izvršavate kupnju ili nas kontaktirate</li>
    <li><strong>Automatiziranih tehnologija:</strong> dok navigirate našom web stranicom, automatski možemo prikupljati tehničke podatke i podatke o korištenju putem kolačića, serverskih zapisa i sličnih tehnologija</li>
    <li><strong>Treće strane:</strong> možemo primiti podatke od pružatelja analitičkih usluga, oglašivačkih mreža i platformi društvenih medija</li>
</ul>

<h2>5. Pravna osnova za obradu</h2>
<p>Vaše osobne podatke obrađujemo na temelju jedne ili više sljedećih pravnih osnova prema članku 6. stavku 1. GDPR-a:</p>
<ul>
    <li><strong>Privola (čl. 6(1)(a)):</strong> dali ste jasnu privolu za obradu vaših osobnih podataka u određenu svrhu (npr. marketinški kolačići, newsletteri)</li>
    <li><strong>Ugovor (čl. 6(1)(b)):</strong> obrada je nužna za izvršavanje ugovora s vama ili za poduzimanje koraka na vaš zahtjev prije sklapanja ugovora</li>
    <li><strong>Zakonska obveza (čl. 6(1)(c)):</strong> obrada je nužna za ispunjavanje zakonske obveze</li>
    <li><strong>Legitimni interes (čl. 6(1)(f)):</strong> obrada je nužna za naše legitimne interese ili interese treće strane, pod uvjetom da vaša prava ne nadjačavaju te interese</li>
</ul>

<h2>6. Kako koristimo vaše podatke</h2>
<p>Prikupljene podatke koristimo u sljedeće svrhe:</p>
<ul>
    <li>Za pružanje, upravljanje i održavanje naše web stranice</li>
    <li>Za poboljšanje i personalizaciju vašeg iskustva</li>
    <li>Za razumijevanje i analizu načina na koji koristite našu web stranicu</li>
    <li>Za komunikaciju s vama, uključujući korisničku podršku, obavijesti i marketing (uz vašu privolu)</li>
    <li>Za obradu transakcija i upravljanje narudžbama</li>
    <li>Za ispunjavanje zakonskih obveza</li>
    <li>Za otkrivanje, sprječavanje i rješavanje tehničkih problema ili prijevara</li>
</ul>

<h2>7. Kolačići i tehnologije praćenja</h2>
<p>Naša web stranica koristi kolačiće i slične tehnologije praćenja. Kolačić je mala tekstualna datoteka koja se postavlja na vaš uređaj kada posjetite web stranicu. Koristimo sljedeće kategorije kolačića:</p>

<h3>7.1 Strogo neophodni kolačići</h3>
<p><?php echo esc_html( $cat_desc_hr['necessary'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['necessary'] ?? array(), 'hr' ); ?>

<h3>7.2 Funkcionalni kolačići</h3>
<p><?php echo esc_html( $cat_desc_hr['functional'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['functional'] ?? array(), 'hr' ); ?>

<h3>7.3 Analitički kolačići</h3>
<p><?php echo esc_html( $cat_desc_hr['analytics'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['analytics'] ?? array(), 'hr' ); ?>

<h3>7.4 Marketinški kolačići</h3>
<p><?php echo esc_html( $cat_desc_hr['marketing'] ); ?></p>
<?php echo self::cookie_table( $cookies_by_cat['marketing'] ?? array(), 'hr' ); ?>

<h3>7.5 Upravljanje postavkama kolačića</h3>
<p>Kada prvi put posjetite našu web stranicu, prikazat će vam se obavijest o kolačićima koja vam omogućuje prihvaćanje ili odbijanje kategorija kolačića. Svoje postavke možete promijeniti u bilo kojem trenutku klikom na ikonu kolačića u donjem kutu stranice ili korištenjem sljedeće poveznice:</p>
<p><a href="#" onclick="if(window.R2CookieConsent){R2CookieConsent.showSettings();}return false;">Upravljanje postavkama kolačića</a></p>
<p>Kolačićima možete upravljati i putem postavki preglednika. Napominjemo da onemogućavanje određenih kolačića može utjecati na funkcionalnost naše web stranice.</p>

<h2>8. Dijeljenje podataka i treće strane</h2>
<p>Vaše osobne podatke možemo dijeliti sa sljedećim kategorijama primatelja:</p>
<ul>
    <li><strong>Pružatelji usluga:</strong> hosting, analitika, email marketing, obrada plaćanja</li>
    <li><strong>Oglašivački partneri:</strong> samo uz vašu izričitu privolu za marketinške kolačiće</li>
    <li><strong>Pravosudna tijela:</strong> kada to zahtijeva zakon ili radi zaštite naših prava</li>
</ul>
<p>Ne prodajemo vaše osobne podatke trećim stranama.</p>

<h2>9. Međunarodni prijenosi podataka</h2>
<p>Neki od naših pružatelja usluga trećih strana mogu se nalaziti izvan Europskog gospodarskog prostora (EGP). Pri prijenosu podataka izvan EGP-a osiguravamo odgovarajuće zaštitne mjere, kao što su Standardne ugovorne klauzule (SUK) koje je odobrila Europska komisija, ili prijenos u zemlje s odgovarajućom razinom zaštite podataka.</p>

<h2>10. Zadržavanje podataka</h2>
<p>Vaše osobne podatke čuvamo samo onoliko dugo koliko je potrebno za ispunjenje svrha za koje su prikupljeni, uključujući ispunjavanje zakonskih, računovodstvenih ili izvještajnih zahtjeva. Zapisi o privoli za kolačiće čuvaju se za vrijeme trajanja privole, nakon čega se automatski brišu.</p>

<h2>11. Vaša prava prema GDPR-u</h2>
<p>Prema GDPR-u imate sljedeća prava u vezi s vašim osobnim podacima:</p>
<ul>
    <li><strong>Pravo pristupa (čl. 15):</strong> zatražiti kopiju vaših osobnih podataka</li>
    <li><strong>Pravo na ispravak (čl. 16):</strong> zatražiti ispravak netočnih podataka</li>
    <li><strong>Pravo na brisanje (čl. 17):</strong> zatražiti brisanje vaših podataka ("pravo na zaborav")</li>
    <li><strong>Pravo na ograničenje obrade (čl. 18):</strong> zatražiti ograničenje obrade podataka</li>
    <li><strong>Pravo na prenosivost podataka (čl. 20):</strong> primiti vaše podatke u strukturiranom, strojno čitljivom formatu</li>
    <li><strong>Pravo na prigovor (čl. 21):</strong> uložiti prigovor na obradu na temelju legitimnog interesa ili izravnog marketinga</li>
    <li><strong>Pravo na povlačenje privole (čl. 7):</strong> povući svoju privolu u bilo kojem trenutku bez utjecaja na zakonitost prethodne obrade</li>
</ul>
<p>Za ostvarivanje bilo kojeg od ovih prava, molimo kontaktirajte nas na: <a href="mailto:<?php echo $dpo; ?>"><?php echo $dpo; ?></a></p>
<p>Također imate pravo podnijeti pritužbu nadležnom tijelu za zaštitu podataka. U Republici Hrvatskoj to je Agencija za zaštitu osobnih podataka (AZOP), <a href="https://azop.hr" target="_blank" rel="noopener">azop.hr</a>.</p>

<h2>12. Sigurnost podataka</h2>
<p>Primjenjujemo odgovarajuće tehničke i organizacijske mjere za zaštitu vaših osobnih podataka od neovlaštenog pristupa, izmjene, otkrivanja ili uništenja. Te mjere uključuju enkripciju, kontrolu pristupa i redovite sigurnosne procjene.</p>

<h2>13. Privatnost djece</h2>
<p>Naša web stranica nije namijenjena djeci mlađoj od 16 godina. Svjesno ne prikupljamo osobne podatke djece. Ako saznamo da smo prikupili podatke djeteta mlađeg od 16 godina bez roditeljske privole, poduzet ćemo korake za njihovo brisanje.</p>

<h2>14. Promjene ove Politike</h2>
<p>Ovu Politiku privatnosti možemo povremeno ažurirati. Sve promjene bit će objavljene na ovoj stranici s ažuriranim datumom revizije. Potičemo vas da povremeno pregledate ovu stranicu.</p>

<h2>15. Kontakt</h2>
<p>Ako imate pitanja o ovoj Politici privatnosti ili načinu na koji postupamo s vašim osobnim podacima, kontaktirajte nas:</p>
<p><strong><?php echo $company; ?></strong><br>
E-pošta: <a href="mailto:<?php echo $email; ?>"><?php echo $email; ?></a><br>
Web: <a href="<?php echo $url; ?>"><?php echo $site; ?></a></p>
<?php
        return ob_get_clean();
    }
}
