<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Shortcodes for RISE2 Cookie Consent
 *
 * [r2cc_settings_button]   - Button to open cookie settings panel
 * [r2cc_cookie_table]      - Full cookie declaration table
 * [r2cc_consent_status]    - Shows current user consent status
 */
class R2CC_Shortcodes {

    public function __construct() {
        add_shortcode( 'r2cc_settings_button', array( $this, 'settings_button' ) );
        add_shortcode( 'r2cc_cookie_table',    array( $this, 'cookie_table' ) );
        add_shortcode( 'r2cc_consent_status',  array( $this, 'consent_status' ) );
    }

    /**
     * [r2cc_settings_button] - Open cookie preferences
     *
     * Attributes:
     *   text   - Button label (default: "Cookie Settings")
     *   class  - Additional CSS classes
     *   style  - "button" | "link" (default: "button")
     */
    public function settings_button( $atts ) {
        $atts = shortcode_atts( array(
            'text'  => Rise2_Cookie_Consent::get_setting( 'btn_settings_label', __( 'Cookie Settings', 'rise2-cookie-consent' ) ),
            'class' => '',
            'style' => 'button',
        ), $atts, 'r2cc_settings_button' );

        $tag = $atts['style'] === 'link' ? 'a' : 'button';
        $classes = 'r2cc-shortcode-settings-btn';
        if ( $atts['class'] ) {
            $classes .= ' ' . esc_attr( $atts['class'] );
        }

        $href = $tag === 'a' ? ' href="#"' : ' type="button"';

        return sprintf(
            '<%1$s%2$s class="%3$s" onclick="if(window.R2CookieConsent){R2CookieConsent.showSettings();}return false;">%4$s</%1$s>',
            $tag,
            $href,
            esc_attr( $classes ),
            esc_html( $atts['text'] )
        );
    }

    /**
     * [r2cc_cookie_table] - Full cookie declaration
     *
     * Attributes:
     *   category - Limit to a specific category slug (default: all)
     *   show_category - Show category column (default: true if showing all)
     */
    public function cookie_table( $atts ) {
        $atts = shortcode_atts( array(
            'category'      => '',
            'show_category' => '',
        ), $atts, 'r2cc_cookie_table' );

        $categories = R2CC_Cookie_Categories::get_categories();
        $lang = R2CC_I18n::get_current_language();

        if ( ! empty( $atts['category'] ) && isset( $categories[ $atts['category'] ] ) ) {
            $categories = array( $atts['category'] => $categories[ $atts['category'] ] );
        }

        $show_cat_col = ( $atts['show_category'] !== '' )
            ? filter_var( $atts['show_category'], FILTER_VALIDATE_BOOLEAN )
            : ( empty( $atts['category'] ) );

        $html = '<div class="r2cc-cookie-declaration">';

        foreach ( $categories as $slug => $cat ) {
            $cookies = R2CC_Cookie_Database::get_cookies( $slug );

            $label = R2CC_I18n::translate( $cat['label'], 'category_label' );
            $desc  = R2CC_I18n::translate( $cat['description'], 'category_desc' );

            $html .= '<div class="r2cc-declaration-category">';
            $html .= '<h3 class="r2cc-declaration-heading">' . esc_html( $label ) . '</h3>';
            $html .= '<p class="r2cc-declaration-desc">' . esc_html( $desc ) . '</p>';

            if ( empty( $cookies ) ) {
                $html .= '<p><em>' . esc_html__( 'No cookies registered in this category.', 'rise2-cookie-consent' ) . '</em></p>';
            } else {
                $html .= '<table class="r2cc-declaration-table">';
                $html .= '<thead><tr>';
                $html .= '<th>' . esc_html__( 'Cookie', 'rise2-cookie-consent' ) . '</th>';
                if ( $show_cat_col ) {
                    $html .= '<th>' . esc_html__( 'Category', 'rise2-cookie-consent' ) . '</th>';
                }
                $html .= '<th>' . esc_html__( 'Domain', 'rise2-cookie-consent' ) . '</th>';
                $html .= '<th>' . esc_html__( 'Duration', 'rise2-cookie-consent' ) . '</th>';
                $html .= '<th>' . esc_html__( 'Description', 'rise2-cookie-consent' ) . '</th>';
                $html .= '</tr></thead><tbody>';

                foreach ( $cookies as $c ) {
                    $html .= '<tr>';
                    $html .= '<td><code>' . esc_html( $c->cookie_name ) . '</code></td>';
                    if ( $show_cat_col ) {
                        $html .= '<td>' . esc_html( $label ) . '</td>';
                    }
                    $html .= '<td>' . esc_html( $c->domain ?: '—' ) . '</td>';
                    $html .= '<td>' . esc_html( $c->duration ?: '—' ) . '</td>';
                    $html .= '<td>' . esc_html( $c->description ?: '—' ) . '</td>';
                    $html .= '</tr>';
                }

                $html .= '</tbody></table>';
            }

            $html .= '</div>';
        }

        $html .= '</div>';
        return $html;
    }

    /**
     * [r2cc_consent_status] - Shows current user consent state
     *
     * Entirely JS-driven since consent is stored in cookies (cache-safe)
     */
    public function consent_status( $atts ) {
        $atts = shortcode_atts( array(
            'show_categories' => 'true',
            'show_change'     => 'true',
        ), $atts, 'r2cc_consent_status' );

        $show_cats   = filter_var( $atts['show_categories'], FILTER_VALIDATE_BOOLEAN );
        $show_change = filter_var( $atts['show_change'], FILTER_VALIDATE_BOOLEAN );

        $categories_json = wp_json_encode( R2CC_Cookie_Categories::get_categories() );

        ob_start();
        ?>
        <div class="r2cc-consent-status" id="r2cc-consent-status-widget">
            <noscript><?php esc_html_e( 'JavaScript is required to view your consent status.', 'rise2-cookie-consent' ); ?></noscript>
        </div>
        <script>
        (function(){
            var w = document.getElementById("r2cc-consent-status-widget");
            if (!w) return;
            function render() {
                if (typeof R2CookieConsent === "undefined") return;
                var consent = R2CookieConsent.getConsent();
                var cats = <?php echo $categories_json; ?>;
                var h = "";
                if (!consent) {
                    h += '<p class="r2cc-status-none"><?php echo esc_js( __( 'No consent recorded yet.', 'rise2-cookie-consent' ) ); ?></p>';
                } else {
                    h += '<p class="r2cc-status-date"><strong><?php echo esc_js( __( 'Consent given:', 'rise2-cookie-consent' ) ); ?></strong> ' + new Date(consent.timestamp).toLocaleDateString() + '</p>';
                    h += '<p class="r2cc-status-type"><strong><?php echo esc_js( __( 'Type:', 'rise2-cookie-consent' ) ); ?></strong> ' + consent.type.replace(/_/g, " ") + '</p>';
                    <?php if ( $show_cats ) : ?>
                    h += '<ul class="r2cc-status-cats">';
                    Object.keys(cats).forEach(function(slug) {
                        var ok = consent.categories && consent.categories.indexOf(slug) !== -1;
                        h += '<li>' + (ok ? '\u2705' : '\u274C') + ' ' + cats[slug].label + '</li>';
                    });
                    h += '</ul>';
                    <?php endif; ?>
                }
                <?php if ( $show_change ) : ?>
                h += '<p><a href="#" onclick="if(window.R2CookieConsent){R2CookieConsent.showSettings();}return false;"><?php echo esc_js( __( 'Change preferences', 'rise2-cookie-consent' ) ); ?></a></p>';
                <?php endif; ?>
                w.innerHTML = h;
            }
            if (document.readyState === "loading") {
                document.addEventListener("DOMContentLoaded", render);
            } else {
                render();
            }
        })();
        </script>
        <?php
        return ob_get_clean();
    }
}
