<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * RISE2 Self-Hosted Plugin Updater
 *
 * Drop-in class for any RISE2 Studio plugin to check for updates
 * from plugins.rise2.studio. Hooks into WordPress's native
 * update_plugins transient so updates appear in Dashboard → Updates.
 *
 * Usage in your main plugin file:
 *
 *   new R2_Plugin_Updater( array(
 *       'slug'        => 'rise2-cookie-consent',
 *       'version'     => '2.0.0',
 *       'basename'    => plugin_basename( __FILE__ ),
 *       'update_url'  => 'https://plugins.rise2.studio/rise2-cookie-consent/info.json',
 *   ) );
 *
 * Server JSON format (info.json):
 *   {
 *     "name": "RISE2 Cookie Consent",
 *     "slug": "rise2-cookie-consent",
 *     "version": "2.1.0",
 *     "requires": "5.8",
 *     "tested": "6.7",
 *     "requires_php": "7.4",
 *     "author": "RISE2 Studio",
 *     "download_url": "https://plugins.rise2.studio/rise2-cookie-consent/rise2-cookie-consent-2.1.0.zip",
 *     "sections": {
 *       "description": "...",
 *       "changelog": "..."
 *     },
 *     "banners": {
 *       "low": "https://plugins.rise2.studio/rise2-cookie-consent/banner-772x250.jpg",
 *       "high": "https://plugins.rise2.studio/rise2-cookie-consent/banner-1544x500.jpg"
 *     },
 *     "icons": {
 *       "1x": "https://plugins.rise2.studio/rise2-cookie-consent/icon-128x128.png",
 *       "2x": "https://plugins.rise2.studio/rise2-cookie-consent/icon-256x256.png"
 *     }
 *   }
 *
 * @version 1.0.0
 */
class R2_Plugin_Updater {

    /** @var string Plugin slug */
    private $slug;

    /** @var string Current installed version */
    private $version;

    /** @var string Plugin basename (e.g. rise2-cookie-consent/rise2-cookie-consent.php) */
    private $basename;

    /** @var string URL to the JSON metadata file */
    private $update_url;

    /** @var string Transient cache key */
    private $cache_key;

    /** @var int Cache duration in seconds (default: 6 hours) */
    private $cache_ttl;

    /** @var object|null Cached remote data */
    private $remote_data = null;

    /**
     * @param array $args {
     *     @type string $slug       Plugin directory slug
     *     @type string $version    Current version
     *     @type string $basename   plugin_basename(__FILE__)
     *     @type string $update_url URL to info.json
     *     @type int    $cache_ttl  Cache duration in seconds (default 21600 = 6h)
     * }
     */
    public function __construct( $args ) {
        $this->slug       = $args['slug'];
        $this->version    = $args['version'];
        $this->basename   = $args['basename'];
        $this->update_url = $args['update_url'];
        $this->cache_ttl  = $args['cache_ttl'] ?? 21600;
        $this->cache_key  = 'r2_updater_' . $this->slug;

        // Hook into WP update system
        add_filter( 'plugins_api',                array( $this, 'plugin_info' ), 20, 3 );
        add_filter( 'site_transient_update_plugins', array( $this, 'check_update' ) );
        add_action( 'upgrader_process_complete',   array( $this, 'clear_cache' ), 10, 2 );

        // Add "Check for updates" link on plugins page
        add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );

        // Handle manual update check
        add_action( 'admin_init', array( $this, 'handle_manual_check' ) );
    }

    /**
     * Fetch remote plugin info (cached)
     *
     * @param bool $force_refresh Bypass cache
     * @return object|false
     */
    private function get_remote_data( $force_refresh = false ) {
        if ( $this->remote_data !== null && ! $force_refresh ) {
            return $this->remote_data;
        }

        // Check transient cache
        if ( ! $force_refresh ) {
            $cached = get_transient( $this->cache_key );
            if ( $cached !== false ) {
                $this->remote_data = $cached;
                return $cached;
            }
        }

        // Fetch from server
        $response = wp_remote_get( $this->update_url, array(
            'timeout'    => 15,
            'sslverify'  => true,
            'user-agent' => 'WordPress/' . get_bloginfo( 'version' ) . '; ' . home_url(),
            'headers'    => array(
                'Accept' => 'application/json',
            ),
        ) );

        if ( is_wp_error( $response ) ) {
            // Cache the failure briefly (5 min) to avoid hammering server
            set_transient( $this->cache_key, false, 300 );
            $this->remote_data = false;
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        if ( $code !== 200 ) {
            set_transient( $this->cache_key, false, 300 );
            $this->remote_data = false;
            return false;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body );

        if ( empty( $data ) || ! isset( $data->version ) ) {
            set_transient( $this->cache_key, false, 300 );
            $this->remote_data = false;
            return false;
        }

        // Cache successful response
        set_transient( $this->cache_key, $data, $this->cache_ttl );
        $this->remote_data = $data;
        return $data;
    }

    /**
     * Filter: Inject update info into WP's update_plugins transient
     *
     * This is what makes the "Update available" notice appear.
     */
    public function check_update( $transient ) {
        if ( empty( $transient->checked ) ) {
            return $transient;
        }

        $remote = $this->get_remote_data();
        if ( ! $remote || ! isset( $remote->version ) ) {
            return $transient;
        }

        // Compare versions
        if ( version_compare( $this->version, $remote->version, '<' ) ) {
            $update = new stdClass();
            $update->slug        = $this->slug;
            $update->plugin      = $this->basename;
            $update->new_version = $remote->version;
            $update->tested      = $remote->tested ?? '';
            $update->requires    = $remote->requires ?? '';
            $update->requires_php = $remote->requires_php ?? '';
            $update->url         = $remote->homepage ?? '';
            $update->package     = $remote->download_url ?? '';

            // Icons for the update screen
            if ( ! empty( $remote->icons ) ) {
                $update->icons = (array) $remote->icons;
            }

            // Banners
            if ( ! empty( $remote->banners ) ) {
                $update->banners = (array) $remote->banners;
            }

            $transient->response[ $this->basename ] = $update;
        } else {
            // Tell WP this plugin is up to date (prevents false positives from wp.org)
            $no_update = new stdClass();
            $no_update->slug        = $this->slug;
            $no_update->plugin      = $this->basename;
            $no_update->new_version = $this->version;
            $no_update->url         = '';
            $no_update->package     = '';
            $transient->no_update[ $this->basename ] = $no_update;
        }

        return $transient;
    }

    /**
     * Filter: Provide full plugin info for the "View details" popup
     */
    public function plugin_info( $result, $action, $args ) {
        if ( $action !== 'plugin_information' ) {
            return $result;
        }

        if ( ! isset( $args->slug ) || $args->slug !== $this->slug ) {
            return $result;
        }

        $remote = $this->get_remote_data();
        if ( ! $remote ) {
            return $result;
        }

        $info = new stdClass();
        $info->name          = $remote->name ?? $this->slug;
        $info->slug          = $this->slug;
        $info->version       = $remote->version;
        $info->author        = $remote->author ?? '';
        $info->author_profile = $remote->author_profile ?? '';
        $info->homepage      = $remote->homepage ?? '';
        $info->requires      = $remote->requires ?? '';
        $info->tested        = $remote->tested ?? '';
        $info->requires_php  = $remote->requires_php ?? '';
        $info->download_link = $remote->download_url ?? '';
        $info->last_updated  = $remote->last_updated ?? '';

        // Sections (description, changelog, FAQ, etc.)
        if ( ! empty( $remote->sections ) ) {
            $info->sections = (array) $remote->sections;
        }

        // Banners
        if ( ! empty( $remote->banners ) ) {
            $info->banners = (array) $remote->banners;
        }

        // Icons
        if ( ! empty( $remote->icons ) ) {
            $info->icons = (array) $remote->icons;
        }

        return $info;
    }

    /**
     * Add "Check for updates" link on the Plugins page
     */
    public function plugin_row_meta( $links, $file ) {
        if ( $file !== $this->basename ) {
            return $links;
        }

        $check_url = wp_nonce_url(
            add_query_arg( 'r2_force_check', $this->slug, admin_url( 'plugins.php' ) ),
            'r2_force_check_' . $this->slug
        );

        $links[] = '<a href="' . esc_url( $check_url ) . '">' . esc_html__( 'Check for updates', 'rise2-cookie-consent' ) . '</a>';

        return $links;
    }

    /**
     * Handle manual "Check for updates" click
     */
    public function handle_manual_check() {
        if ( ! isset( $_GET['r2_force_check'] ) || $_GET['r2_force_check'] !== $this->slug ) {
            return;
        }

        if ( ! current_user_can( 'update_plugins' ) ) {
            return;
        }

        check_admin_referer( 'r2_force_check_' . $this->slug );

        // Clear caches
        delete_transient( $this->cache_key );
        delete_site_transient( 'update_plugins' );

        // Redirect back to plugins page with notice
        wp_safe_redirect( add_query_arg( 'r2_checked', $this->slug, admin_url( 'plugins.php' ) ) );
        exit;
    }

    /**
     * Clear cache after plugin update completes
     */
    public function clear_cache( $upgrader, $options ) {
        if ( $options['action'] === 'update' && $options['type'] === 'plugin' ) {
            $plugins = $options['plugins'] ?? array();
            if ( in_array( $this->basename, $plugins, true ) ) {
                delete_transient( $this->cache_key );
            }
        }
    }
}
