<?php
/**
 * Plugin Name: RISE2 Cookie Consent
 * Plugin URI: https://rise2.studio
 * Description: A powerful, standalone GDPR/ePrivacy cookie consent plugin with automatic cookie scanning, predefined categories, customizable banners, and granular consent management. No third-party services required.
 * Version: 2.0.4
 * Author: RISE2 Studio
 * Author URI: https://rise2.studio
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: rise2-cookie-consent
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin constants
define( 'R2CC_VERSION', '2.0.4' );
define( 'R2CC_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'R2CC_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'R2CC_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main Plugin Class
 */
final class Rise2_Cookie_Consent {

    private static $instance = null;

    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->includes();
        $this->init_hooks();
    }

    private function includes() {
        require_once R2CC_PLUGIN_DIR . 'includes/class-cookie-database.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-cookie-scanner.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-cookie-categories.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-consent-logger.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-i18n.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-google-consent-mode.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-privacy-policy.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-auto-blocker.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-shortcodes.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-cron.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-updater.php';
        require_once R2CC_PLUGIN_DIR . 'includes/class-frontend.php';

        if ( is_admin() ) {
            require_once R2CC_PLUGIN_DIR . 'admin/class-admin.php';
        }
    }

    private function init_hooks() {
        register_activation_hook( __FILE__, array( $this, 'activate' ) );
        register_deactivation_hook( __FILE__, array( $this, 'deactivate' ) );
        register_uninstall_hook( __FILE__, array( __CLASS__, 'uninstall' ) );

        add_action( 'init', array( $this, 'load_textdomain' ) );
        add_action( 'admin_init', array( $this, 'maybe_upgrade' ) );
        add_action( 'wp_ajax_r2cc_save_consent', array( $this, 'ajax_save_consent' ) );
        add_action( 'wp_ajax_nopriv_r2cc_save_consent', array( $this, 'ajax_save_consent' ) );
        add_action( 'wp_ajax_r2cc_scan_cookies', array( $this, 'ajax_scan_cookies' ) );
        add_action( 'wp_ajax_r2cc_scan_browser_cookies', array( $this, 'ajax_scan_browser_cookies' ) );
        add_action( 'wp_ajax_r2cc_save_cookie', array( $this, 'ajax_save_cookie' ) );
        add_action( 'wp_ajax_r2cc_delete_cookie', array( $this, 'ajax_delete_cookie' ) );
        add_action( 'wp_ajax_r2cc_reset_settings', array( $this, 'ajax_reset_settings' ) );
        add_action( 'wp_ajax_r2cc_generate_privacy_policy', array( $this, 'ajax_generate_privacy_policy' ) );
        add_action( 'wp_ajax_r2cc_preview_privacy_policy', array( $this, 'ajax_preview_privacy_policy' ) );
        add_action( 'wp_ajax_r2cc_export_settings', array( $this, 'ajax_export_settings' ) );
        add_action( 'wp_ajax_r2cc_import_settings', array( $this, 'ajax_import_settings' ) );
        add_action( 'wp_ajax_r2cc_get_analytics', array( $this, 'ajax_get_analytics' ) );

        // Initialize modules
        R2CC_I18n::init();
        new R2CC_Google_Consent_Mode();
        new R2CC_Auto_Blocker();
        new R2CC_Shortcodes();
        new R2CC_Cron();

        // Self-hosted update checker
        new R2_Plugin_Updater( array(
            'slug'       => 'rise2-cookie-consent',
            'version'    => R2CC_VERSION,
            'basename'   => R2CC_PLUGIN_BASENAME,
            'update_url' => 'https://plugins.rise2.studio/rise2-cookie-consent/info.json',
        ) );
    }

    public function load_textdomain() {
        load_plugin_textdomain( 'rise2-cookie-consent', false, dirname( R2CC_PLUGIN_BASENAME ) . '/languages' );
    }

    public function activate() {
        R2CC_Cookie_Database::create_tables();
        R2CC_Cookie_Categories::seed_defaults();

        // Default settings
        $defaults = array(
            'enabled'              => 1,
            'banner_position'      => 'bottom-left', // bottom-left, bottom-right, center-overlay
            'banner_bg_color'      => '#1a1a2e',
            'banner_text_color'    => '#e0e0e0',
            'btn_accept_bg'        => '#4f46e5',
            'btn_accept_text'      => '#ffffff',
            'btn_reject_bg'        => 'transparent',
            'btn_reject_text'      => '#e0e0e0',
            'btn_settings_bg'      => 'transparent',
            'btn_settings_text'    => '#e0e0e0',
            'consent_expiry_days'  => 365,
            'show_reject_all'      => 1,
            'show_powered_by'      => 1,
            'auto_block_scripts'   => 0,
            'banner_title'         => __( 'We value your privacy', 'rise2-cookie-consent' ),
            'banner_description'   => __( 'We use cookies to enhance your browsing experience, serve personalized content, and analyze our traffic. By clicking "Accept All", you consent to our use of cookies. You can customize your preferences by clicking "Cookie Settings".', 'rise2-cookie-consent' ),
            'btn_accept_label'     => __( 'Accept All', 'rise2-cookie-consent' ),
            'btn_reject_label'     => __( 'Reject All', 'rise2-cookie-consent' ),
            'btn_settings_label'   => __( 'Cookie Settings', 'rise2-cookie-consent' ),
            'btn_save_label'       => __( 'Save Preferences', 'rise2-cookie-consent' ),
            'privacy_policy_url'   => '',
            'privacy_policy_label' => __( 'Privacy Policy', 'rise2-cookie-consent' ),
            'custom_css'           => '',
            'geo_restriction'      => 'everywhere', // everywhere, eu-only
            // Google Consent Mode v2
            'gcm_enabled'          => 0,
            'gcm_wait_for_update'  => 500,
            'gcm_ads_data_redaction' => 1,
            'gcm_url_passthrough'  => 0,
            'gcm_region'           => '',
            // Privacy Policy Generator
            'pp_company_name'      => get_bloginfo( 'name' ),
            'pp_company_address'   => '',
            'pp_company_oib'       => '',
            'pp_dpo_email'         => get_option( 'admin_email' ),
            // Consent versioning
            'consent_version'      => '1.0',
            // Scheduled scan
            'scan_frequency'       => 'r2cc_weekly',
            'scan_email_notify'    => 0,
        );

        foreach ( $defaults as $key => $value ) {
            if ( false === get_option( 'r2cc_' . $key ) ) {
                update_option( 'r2cc_' . $key, $value );
            }
        }

        // Schedule cron
        $freq = get_option( 'r2cc_scan_frequency', 'r2cc_weekly' );
        if ( $freq !== 'disabled' ) {
            R2CC_Cron::schedule( $freq );
        }

        flush_rewrite_rules();

        // Store current version for upgrade routines
        update_option( 'r2cc_version', R2CC_VERSION );
    }

    public function deactivate() {
        R2CC_Cron::unschedule();
        flush_rewrite_rules();
    }

    /**
     * Run upgrade routines when plugin version changes.
     */
    public function maybe_upgrade() {
        $installed_version = get_option( 'r2cc_version', '0' );
        if ( version_compare( $installed_version, R2CC_VERSION, '>=' ) ) {
            return;
        }

        // v2.0.4+: Remove admin-only cookies and false positives from database
        if ( version_compare( $installed_version, '2.0.4', '<' ) ) {
            $this->cleanup_admin_cookies();
        }

        update_option( 'r2cc_version', R2CC_VERSION );
    }

    /**
     * Remove admin-only cookies that regular visitors never encounter.
     */
    private function cleanup_admin_cookies() {
        global $wpdb;
        $table = $wpdb->prefix . 'r2cc_cookies';

        $admin_patterns = array(
            'wordpress_logged_in_%',
            'wordpress_sec_%',
            'wordpress_test_cookie',
            'wp-settings-%',
            'wp-settings-time-%',
            'comment_author_%',
            'comment_author_email_%',
            'yoast-seo-%',
        );

        // Also remove false positives from the old grouped form plugin scan
        $false_positive_patterns = array(
            'wp_cf7_%',   // Only added if CF7 is actually active now
            'wpforms_%',  // Only added if WPForms is actually active now
        );

        $all_patterns = array_merge( $admin_patterns, $false_positive_patterns );

        foreach ( $all_patterns as $pattern ) {
            $wpdb->query(
                $wpdb->prepare( "DELETE FROM {$table} WHERE cookie_name LIKE %s", $pattern )
            );
        }
    }

    public static function uninstall() {
        global $wpdb;

        // Remove all options
        $wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'r2cc_%'" );

        // Drop tables
        $wpdb->query( "DROP TABLE IF EXISTS {$wpdb->prefix}r2cc_cookies" );
        $wpdb->query( "DROP TABLE IF EXISTS {$wpdb->prefix}r2cc_consent_logs" );
    }

    /**
     * Get a plugin setting with fallback
     */
    public static function get_setting( $key, $default = '' ) {
        return get_option( 'r2cc_' . $key, $default );
    }

    /**
     * AJAX: Save user consent
     */
    public function ajax_save_consent() {
        check_ajax_referer( 'r2cc_nonce', 'nonce' );

        $categories = isset( $_POST['categories'] ) ? array_map( 'sanitize_text_field', (array) $_POST['categories'] ) : array();
        $consent_type = isset( $_POST['consent_type'] ) ? sanitize_text_field( $_POST['consent_type'] ) : 'custom';

        R2CC_Consent_Logger::log_consent( $categories, $consent_type );

        wp_send_json_success( array( 'message' => 'Consent saved' ) );
    }

    /**
     * AJAX: Trigger cookie scan
     */
    public function ajax_scan_cookies() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $scanner = new R2CC_Cookie_Scanner();
        $results = $scanner->scan();

        wp_send_json_success( $results );
    }

    /**
     * AJAX: Process real cookies detected by the admin's browser.
     */
    public function ajax_scan_browser_cookies() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $cookies = isset( $_POST['cookies'] ) ? array_map( 'sanitize_text_field', (array) $_POST['cookies'] ) : array();

        $scanner = new R2CC_Cookie_Scanner();
        $results = $scanner->process_browser_cookies( $cookies );

        wp_send_json_success( $results );
    }

    /**
     * AJAX: Save/update a cookie entry
     */
    public function ajax_save_cookie() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $data = array(
            'cookie_name'  => sanitize_text_field( $_POST['cookie_name'] ?? '' ),
            'category'     => sanitize_text_field( $_POST['category'] ?? 'necessary' ),
            'domain'       => sanitize_text_field( $_POST['domain'] ?? '' ),
            'duration'     => sanitize_text_field( $_POST['duration'] ?? '' ),
            'description'  => sanitize_textarea_field( $_POST['description'] ?? '' ),
            'is_regex'     => intval( $_POST['is_regex'] ?? 0 ),
            'script_url'   => esc_url_raw( $_POST['script_url'] ?? '' ),
        );

        $id = intval( $_POST['cookie_id'] ?? 0 );

        $result = R2CC_Cookie_Database::save_cookie( $data, $id );

        if ( $result ) {
            wp_send_json_success( array( 'message' => 'Cookie saved', 'id' => $result ) );
        } else {
            wp_send_json_error( 'Failed to save cookie' );
        }
    }

    /**
     * AJAX: Delete a cookie entry
     */
    public function ajax_delete_cookie() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $id = intval( $_POST['cookie_id'] ?? 0 );

        if ( R2CC_Cookie_Database::delete_cookie( $id ) ) {
            wp_send_json_success( 'Deleted' );
        } else {
            wp_send_json_error( 'Failed to delete' );
        }
    }

    /**
     * AJAX: Reset settings to defaults
     */
    public function ajax_reset_settings() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        global $wpdb;
        $wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE 'r2cc_%'" );

        $this->activate();

        wp_send_json_success( 'Settings reset' );
    }

    /**
     * AJAX: Generate privacy policy page
     */
    public function ajax_generate_privacy_policy() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $lang = sanitize_text_field( $_POST['lang'] ?? 'en' );
        $data = array(
            'company_name'    => sanitize_text_field( $_POST['company_name'] ?? '' ),
            'company_address' => sanitize_text_field( $_POST['company_address'] ?? '' ),
            'company_oib'     => sanitize_text_field( $_POST['company_oib'] ?? '' ),
            'contact_email'   => sanitize_email( $_POST['contact_email'] ?? '' ),
            'dpo_email'       => sanitize_email( $_POST['dpo_email'] ?? '' ),
        );

        // Save fields for reuse
        update_option( 'r2cc_pp_company_name', $data['company_name'] );
        update_option( 'r2cc_pp_company_address', $data['company_address'] );
        update_option( 'r2cc_pp_company_oib', $data['company_oib'] );
        update_option( 'r2cc_pp_dpo_email', $data['dpo_email'] );

        $page_id = R2CC_Privacy_Policy_Generator::create_page( $lang, $data );

        if ( $page_id && ! is_wp_error( $page_id ) ) {
            wp_send_json_success( array(
                'page_id'  => $page_id,
                'edit_url' => get_edit_post_link( $page_id, 'raw' ),
                'view_url' => get_permalink( $page_id ),
                'message'  => __( 'Privacy policy page created as draft. Review and publish it.', 'rise2-cookie-consent' ),
            ) );
        } else {
            wp_send_json_error( 'Failed to create page.' );
        }
    }

    /**
     * AJAX: Preview privacy policy content
     */
    public function ajax_preview_privacy_policy() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $lang = sanitize_text_field( $_POST['lang'] ?? 'en' );
        $data = array(
            'company_name'    => sanitize_text_field( $_POST['company_name'] ?? '' ),
            'company_address' => sanitize_text_field( $_POST['company_address'] ?? '' ),
            'company_oib'     => sanitize_text_field( $_POST['company_oib'] ?? '' ),
            'contact_email'   => sanitize_email( $_POST['contact_email'] ?? '' ),
            'dpo_email'       => sanitize_email( $_POST['dpo_email'] ?? '' ),
        );

        $html = R2CC_Privacy_Policy_Generator::generate( $lang, $data );

        wp_send_json_success( array( 'html' => $html ) );
    }

    /**
     * AJAX: Export all settings as JSON
     */
    public function ajax_export_settings() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        global $wpdb;

        // Gather all r2cc_ options
        $options = $wpdb->get_results(
            "SELECT option_name, option_value FROM {$wpdb->options} WHERE option_name LIKE 'r2cc_%'",
            ARRAY_A
        );

        $settings = array();
        foreach ( $options as $opt ) {
            $settings[ $opt['option_name'] ] = $opt['option_value'];
        }

        // Gather cookies
        $cookies = $wpdb->get_results(
            "SELECT cookie_name, category, domain, duration, description, is_regex, script_url
             FROM {$wpdb->prefix}r2cc_cookies",
            ARRAY_A
        );

        $export = array(
            'plugin'   => 'rise2-cookie-consent',
            'version'  => R2CC_VERSION,
            'exported' => current_time( 'mysql' ),
            'site_url' => home_url(),
            'settings' => $settings,
            'cookies'  => $cookies,
        );

        wp_send_json_success( $export );
    }

    /**
     * AJAX: Import settings from JSON
     */
    public function ajax_import_settings() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $json = isset( $_POST['import_data'] ) ? wp_unslash( $_POST['import_data'] ) : '';
        $data = json_decode( $json, true );

        if ( ! $data || ! isset( $data['plugin'] ) || $data['plugin'] !== 'rise2-cookie-consent' ) {
            wp_send_json_error( __( 'Invalid import file. Must be a RISE2 Cookie Consent export.', 'rise2-cookie-consent' ) );
        }

        $imported_settings = 0;
        $imported_cookies  = 0;

        // Import settings
        if ( ! empty( $data['settings'] ) && is_array( $data['settings'] ) ) {
            foreach ( $data['settings'] as $name => $value ) {
                // Only import r2cc_ prefixed options
                if ( strpos( $name, 'r2cc_' ) === 0 ) {
                    update_option( $name, $value );
                    $imported_settings++;
                }
            }
        }

        // Import cookies
        if ( ! empty( $data['cookies'] ) && is_array( $data['cookies'] ) ) {
            foreach ( $data['cookies'] as $cookie ) {
                $cookie_data = array(
                    'cookie_name' => sanitize_text_field( $cookie['cookie_name'] ?? '' ),
                    'category'    => sanitize_text_field( $cookie['category'] ?? 'necessary' ),
                    'domain'      => sanitize_text_field( $cookie['domain'] ?? '' ),
                    'duration'    => sanitize_text_field( $cookie['duration'] ?? '' ),
                    'description' => sanitize_textarea_field( $cookie['description'] ?? '' ),
                    'is_regex'    => intval( $cookie['is_regex'] ?? 0 ),
                    'script_url'  => esc_url_raw( $cookie['script_url'] ?? '' ),
                );

                if ( ! empty( $cookie_data['cookie_name'] ) ) {
                    R2CC_Cookie_Database::save_cookie( $cookie_data );
                    $imported_cookies++;
                }
            }
        }

        // Reschedule cron with potentially new frequency
        $freq = get_option( 'r2cc_scan_frequency', 'r2cc_weekly' );
        R2CC_Cron::schedule( $freq );

        wp_send_json_success( array(
            'message'  => sprintf(
                __( 'Import complete: %d settings, %d cookies imported.', 'rise2-cookie-consent' ),
                $imported_settings,
                $imported_cookies
            ),
            'settings' => $imported_settings,
            'cookies'  => $imported_cookies,
        ) );
    }

    /**
     * AJAX: Get analytics data for dashboard
     */
    public function ajax_get_analytics() {
        check_ajax_referer( 'r2cc_admin_nonce', 'nonce' );

        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Unauthorized' );
        }

        $days = intval( $_POST['days'] ?? 30 );
        $days = max( 7, min( 90, $days ) );

        wp_send_json_success( array(
            'daily'          => R2CC_Consent_Logger::get_daily_counts( $days ),
            'category_rates' => R2CC_Consent_Logger::get_category_rates(),
            'type_breakdown' => R2CC_Consent_Logger::get_type_breakdown(),
            'stats'          => R2CC_Consent_Logger::get_stats(),
        ) );
    }
}

// Initialize plugin
function r2cc() {
    return Rise2_Cookie_Consent::get_instance();
}

r2cc();
